package com.finconsgroup.itserr.marketplace.institutional_page.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputModerationStatusDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSearchModerationInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.validation.ValidAssociationToLoad;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.Set;
import java.util.UUID;

/**
 * The Moderation API: it provides endpoints to approve or reject and retrieve InstitutionalPages
 * that have been sent to moderation,
 */
@Tag(name = "Moderation", description = "The Moderation API: it provides endpoints to approve or reject and retrieve pending InstitutionalPages.")
@SecurityRequirement(name = "BearerAuth")
public interface ModerationApi {

    /**
     * Retrieves a paginated list of all InstitutionalPages under moderation (Pending or Rejected), matching the search criteria - moderator only.
     *
     * @param inputSearchModerationInstitutionalPageDto the dto containing the filters to be applied
     * @param associationsToLoad                     comma separated list of the associations to be returned (default is "all").
     * @param pageNumber                             the page number to retrieve (default is 0)
     * @param pageSize                               the number of InstitutionalPages per page (default is 10)
     * @param sort                                   the field to sort by (default is "name")
     * @param direction                              the direction of sorting (default is ascending)
     * @return a page of {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Retrieves a paginated list of all InstitutionalPages under moderation (Pending or Rejected), matching the search criteria - moderator only",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @PostMapping(
            value = "/institutional-page/institutional-pages/moderation/search",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    ResponseEntity<OutputPageDto<OutputInstitutionalPageDto>> searchModerationInstitutionalPages(
            @Valid @RequestBody InputSearchModerationInstitutionalPageDto inputSearchModerationInstitutionalPageDto,
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction
    );


    /**
     * Retrieves a paginated list of all InstitutionalPages under moderation (Pending or Rejected) - moderator only.
     *
     * @param associationsToLoad comma separated list of the associations to be returned (default is "all").
     * @param pageNumber the page number
     * @param pageSize the size of the page
     * @param sort the attribute to sort by
     * @param direction the sorting direction (ASC or DESC)
     * @return the pending institutional pages
     */
    @Operation(summary = "Retrieves a paginated list of all InstitutionalPages under moderation (Pending or Rejected) - moderator only")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @GetMapping(path = "/institutional-page/institutional-pages/moderation", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<OutputPageDto<OutputInstitutionalPageDto>> getModerationInstitutionalPages(
            @RequestParam(name = "associationsToLoad", defaultValue = "all", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "name", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) SortDirection direction
    );

    /**
     * Retrieves a InstitutionalPage under moderation (Pending or Rejected) by id - moderator only.
     *
     * @param institutionalPageId the id of the institutional page
     * @return the pending institutional page
     */
    @Operation(summary = "Retrieves a InstitutionalPage under moderation (Pending or Rejected) by id - moderator only")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @GetMapping("/institutional-page/institutional-pages/moderation/{institutionalPageId}")
    ResponseEntity<OutputInstitutionalPageDto> getModerationInstitutionalPageById(@PathVariable UUID institutionalPageId);

    /**
     * Approve or reject an institutional page
     *
     * @param institutionalPageId the id of the institutional page
     * @param inputModerationStatusDto the DTO to approve or reject a pending Institutional Page
     * @return the approved/rejected institutional page
     */
    @Operation(summary = "Approve or reject an institutional page")
    @ApiResponses(value = @ApiResponse(responseCode = "200", description = "OK"))
    @PatchMapping("/institutional-page/institutional-pages/moderation/{institutionalPageId}")
    ResponseEntity<OutputInstitutionalPageDto> changeInstitutionalPageModerationStatus(
            @PathVariable UUID institutionalPageId,
            @Valid @RequestBody InputModerationStatusDto inputModerationStatusDto);

}
