package com.finconsgroup.itserr.marketplace.institutional_page.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthenticationException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.InstitutionalPageDmClient;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.CreateParagraphIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.ModerationStatusIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.ParagraphIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.UpdateParagraphIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputParagraphDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.mapper.InstitutionalPageMapper;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.mapper.ParagraphMapper;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.InstitutionalPageProducer;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationMessageFactory;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationProducer;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphCreatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphDeletedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphUpdatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.service.ParagraphService;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import feign.FeignException;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import java.util.NoSuchElementException;
import java.util.UUID;

@Service
@RequiredArgsConstructor
public class DefaultParagraphService implements ParagraphService {
    private final ParagraphMapper paragraphMapper;
    private final InstitutionalPageMapper institutionalPageMapper;

    private final InstitutionalPageDmClient institutionalPageDmClient;
    
    private final NotificationMessageFactory notificationMessageFactory;
    private final NotificationProducer notificationProducer;
    private final InstitutionalPageProducer institutionalPageProducer;

    @Override
    public OutputInstitutionalPageDto createParagraph(UUID institutionalPageId, InputParagraphDto paragraphDto, Integer position) {
        try {
            CreateParagraphIPDmDto paragraphDmDto = paragraphMapper.toCreateParagraphIPDmDto(paragraphDto);
            InstitutionalPageIPDmDto institutionalPageDmDto = institutionalPageDmClient.createParagraph(institutionalPageId, paragraphDmDto, position);
            
            OutputInstitutionalPageDto outputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageDmDto, true);
            
            // Send notification for paragraph creation
            MessagingEventDto<ParagraphCreatedNotificationData> notificationMessage = notificationMessageFactory.createParagraphCreatedNotification(paragraphDto, outputInstitutionalPageDto);
            notificationProducer.publishParagraphCreatedNotification(notificationMessage);

            // produce the message with the entity, related to the approved version, not the updated one.
            InstitutionalPageIPDmDto originalInstitutionalPageDmDto = retrieveApprovedInstitutionalPageById(institutionalPageId);
            OutputInstitutionalPageDto originalOutputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(originalInstitutionalPageDmDto, true);
            institutionalPageProducer.publishUpdateEvent(originalOutputInstitutionalPageDto);

            return outputInstitutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public OutputInstitutionalPageDto updateParagraph(UUID institutionalPageId, UUID paragraphId, InputParagraphDto paragraphDto) {
        try {
            UpdateParagraphIPDmDto paragraphDmDto = paragraphMapper.toUpdateParagraphIPDmDto(paragraphDto);
            InstitutionalPageIPDmDto institutionalPageDmDto = institutionalPageDmClient.updateParagraph(institutionalPageId, paragraphId, paragraphDmDto);
            
            OutputInstitutionalPageDto outputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageDmDto, true);
            
            // Send notification for paragraph update
            MessagingEventDto<ParagraphUpdatedNotificationData> notificationMessage = notificationMessageFactory.createParagraphUpdatedNotification(paragraphDto, outputInstitutionalPageDto);
            notificationProducer.publishParagraphUpdatedNotification(notificationMessage);

            // produce the message with the entity, related to the approved version, not the updated one.
            InstitutionalPageIPDmDto originalInstitutionalPageDmDto = retrieveApprovedInstitutionalPageById(institutionalPageId);
            OutputInstitutionalPageDto originalOutputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(originalInstitutionalPageDmDto, true);
            institutionalPageProducer.publishUpdateEvent(originalOutputInstitutionalPageDto);

            return outputInstitutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e.getMessage());
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    @Override
    public OutputInstitutionalPageDto deleteParagraph(UUID institutionalPageId, UUID paragraphId) {
        try {
            ParagraphIPDmDto paragraphIPDmDto = institutionalPageDmClient.getInstitutionalPageById(institutionalPageId, null, false)
                    .getParagraphs()
                    .stream()
                    .filter(paragraph -> paragraph.getId().equals(paragraphId))
                    .findAny()
                    .orElseThrow();

            InstitutionalPageIPDmDto institutionalPageDmDto = institutionalPageDmClient.deleteParagraph(institutionalPageId, paragraphId);
            
            OutputInstitutionalPageDto outputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(institutionalPageDmDto, true);
            
            // Send notification for paragraph deletion
            MessagingEventDto<ParagraphDeletedNotificationData> notificationMessage = notificationMessageFactory.createParagraphDeletedNotification(
                    InputParagraphDto.builder().title(paragraphIPDmDto.getTitle()).build(),
                    outputInstitutionalPageDto
            );
            notificationProducer.publishParagraphDeletedNotification(notificationMessage);

            // produce the message with the entity, related to the approved version, not the updated one.
            InstitutionalPageIPDmDto originalInstitutionalPageDmDto = retrieveApprovedInstitutionalPageById(institutionalPageId);
            OutputInstitutionalPageDto originalOutputInstitutionalPageDto = institutionalPageMapper.toOutputInstitutionalPageDto(originalInstitutionalPageDmDto, true);
            institutionalPageProducer.publishUpdateEvent(originalOutputInstitutionalPageDto);

            return outputInstitutionalPageDto;
        } catch (FeignException.Unauthorized e) {
            throw new WP2AuthenticationException(e.getMessage());
        } catch (FeignException.NotFound e) {
            throw new WP2ResourceNotFoundException(e.getMessage());
        } catch (NoSuchElementException e) {
            throw new WP2ResourceNotFoundException(paragraphId);
        } catch (FeignException e) {
            throw new WP2BusinessException(e);
        }
    }

    private InstitutionalPageIPDmDto retrieveApprovedInstitutionalPageById(UUID institutionalPageId) {
        return institutionalPageDmClient.getInstitutionalPageById(institutionalPageId, ModerationStatusIPDmDto.APPROVED, false);
    }

}
