package com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.MemberRequestType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.UUID;

/**
 * Entity class representing a join request or invitation
 * to contribute as member of an institutional page.
 * This entity is mapped to the "member_request" table in the database
 *
 * <p>Example usage:
 * <pre>
 * MemberRequestEntity memberRequestEntity = MemberRequestEntity.builder()
 *      .institutionalPageId(UUID.randomUUID())
 *      .userId(UUID.randomUUID())
 *      .requestType("invitation")
 *      .requestStatus("pending")
 *      .build();
 * </pre>
 * </p>
 */
@Entity
@Table(name = "member_request")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class MemberRequestEntity extends AbstractUUIDEntity {

    /**
     * ID of user.
     */
    @Column(name = "user_id", nullable = false)
    private UUID userId;

    /**
     * ID of the institutional page.
     */
    @Column(name = "institutional_page_id", nullable = false)
    private UUID institutionalPageId;

    /**
     * type of request (join request or invitation) .
     */
    @Column(name = "request_type", nullable = false)
    private MemberRequestType requestType;

    /**
     * Indicate if the user should be added as wp leader.
     */
    @Column(name = "wp_leader", nullable = false)
    private boolean wpLeader;

    /**
     * invitation, join request or rejection message
     */
    @Column(name = "message")
    private String message;

    /**
     * The timestamp when the institutionalPage was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the institutionalPage was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     * </p>
     */
    @Version
    private long version;

    /**
     * Sets both {@code creationTime} and {@code updateTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @Override
    public void prePersist() {
        super.prePersist();
        Instant currentTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
        if (creationTime == null) {
            creationTime = currentTime;
        }
        updateTime = currentTime;
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
    }

}
