package com.finconsgroup.itserr.marketplace.institutionalpage.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.ArchiveHelper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.DtoBuilder;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.EntityBuilder;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.InstitutionalPageHelper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.InstitutionalPageModerationHandler;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.MemberHelper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputCreateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputPatchIPModerationRequestDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchForMemberInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchModerationInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchPublishedInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputUpdateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputModerationDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputRequestUpdateDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.ModerationOperationType;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.ModerationStatus;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.InstitutionalPageHasChildrenException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.InvalidInputParametersException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.InvalidModerationStatusException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.NotInstitutionalPageWPLeaderException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.UpdateLockedException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.SearchForMemberInstitutionalPageModel;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.model.SearchInstitutionalPageModel;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.service.InstitutionalPageService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * Default implementation of {@link InstitutionalPageService}
 * to perform operations related to institutionalPage resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultInstitutionalPageService implements InstitutionalPageService {

    private final InstitutionalPageModerationHandler institutionalPageModerationHandler;
    private final InstitutionalPageHelper institutionalPageHelper;
    private final MemberHelper memberHelper;
    private final ArchiveHelper archiveHelper;
    private final EntityBuilder entityBuilder;
    private final DtoBuilder dtoBuilder;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto createInstitutionalPage(
            @NonNull UUID userId,
            @NonNull InputCreateInstitutionalPageDto inputCreateInstitutionalPageDto
    ) {
        // if provided from input, retrieves parent institutional page
        UUID parentInstitutionalPageId = inputCreateInstitutionalPageDto.getParentInstitutionalPageId();
        InstitutionalPageEntity parentInstitutionalPageEntity = null;
        if (parentInstitutionalPageId != null) {
            // if parent exists, user must be wpLeader
            memberHelper.verifyWpLeaderOrThrow(userId, parentInstitutionalPageId);
            parentInstitutionalPageEntity = institutionalPageHelper
                    .retrieveOriginalForMemberById(userId, parentInstitutionalPageId);
            if (!parentInstitutionalPageEntity.getModerationStatus().equals(ModerationStatus.APPROVED)) {
                throw new InvalidModerationStatusException(
                        parentInstitutionalPageId,
                        parentInstitutionalPageEntity.getModerationStatus()
                );
            }
        }
        // build entity
        InstitutionalPageEntity institutionalPageEntity = entityBuilder
                .buildInstitutionalPage(userId, parentInstitutionalPageEntity, inputCreateInstitutionalPageDto);
        // moderation process
        institutionalPageEntity = institutionalPageModerationHandler.create(institutionalPageEntity);
        // save institutional page and return output dto
        InstitutionalPageEntity savedInstitutionalPageEntity = institutionalPageHelper.save(userId, institutionalPageEntity);
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPageEntity);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputInstitutionalPageDto> findAllInstitutionalPages(
            @NonNull UUID userId,
            @NonNull ModerationStatus moderationStatus,
            boolean includePublishedAndNotMember,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable
    ) {
        Page<InstitutionalPageEntity> institutionalPageEntityPage = institutionalPageHelper
                .retrieveAllByStatusForMember(userId, moderationStatus, includePublishedAndNotMember, pageable);
        return dtoBuilder.mapEntitiesToDtos(userId, institutionalPageEntityPage, associationsToLoad);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputInstitutionalPageDto> searchInstitutionalPages(
            @NonNull UUID userId,
            @NonNull ModerationStatus moderationStatus,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable,
            @NonNull InputSearchForMemberInstitutionalPageDto inputSearchForMemberInstitutionalPageDto
    ) {
        SearchForMemberInstitutionalPageModel searchInstitutionalPageModel = dtoBuilder
                .toFilters(inputSearchForMemberInstitutionalPageDto);
        Page<InstitutionalPageEntity> institutionalPageEntityPage = institutionalPageHelper
                .retrieveAllByStatusForMemberWithFilter(userId, moderationStatus, pageable, searchInstitutionalPageModel);
        return dtoBuilder.mapEntitiesToDtos(userId, institutionalPageEntityPage, associationsToLoad);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputInstitutionalPageDto findInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            ModerationStatus moderationStatus,
            boolean includePublishedAndNotMember
    ) {
        // includePublishedAndNotMember can be true only when moderationStatus is APPROVED.
        // The user is allowed to retrieve only the APPROVED version of the Institutional Page he is not contributing to.
        if (includePublishedAndNotMember && !ModerationStatus.APPROVED.equals(moderationStatus)) {
            throw new InvalidInputParametersException(
                    "Parameter 'includePublishedAndNotMember' can be true only when 'moderationStatus' is " + ModerationStatus.APPROVED
            );
        }
        InstitutionalPageEntity institutionalPageEntity;
        if (moderationStatus == null) {
            // retrieve latest version of institutional page
            institutionalPageEntity = institutionalPageHelper.retrieveLatestForMemberById(userId, institutionalPageId);
        } else {
            institutionalPageEntity = institutionalPageHelper.retrieveByStatusForMemberById(userId, institutionalPageId, moderationStatus, includePublishedAndNotMember);
        }
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, institutionalPageEntity);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto updateInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputUpdateInstitutionalPageDto inputUpdateInstitutionalPageDto
    ) {
        // 'retrieveLatestForMemberById' already checks membership, but we call 'verifyWpLeaderOrThrow' first
        // to return meaningful errors: if 'institutionalPageId' refers to a published page,
        // it's clearer to return "not a member" instead of "institutional page not found"
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page entity
        InstitutionalPageEntity retrievedInstitutionalPageEntity = institutionalPageHelper
                .retrieveLatestForMemberById(userId, institutionalPageId);
        // if maintainer is set, it must be a wpLeader
        UUID maintainer = inputUpdateInstitutionalPageDto.getMaintainer();
        if (maintainer != null) {
            boolean maintainerIsWPLeader = memberHelper.isWpLeader(maintainer, retrievedInstitutionalPageEntity);
            if (!maintainerIsWPLeader) {
                throw new NotInstitutionalPageWPLeaderException(userId, institutionalPageId);
            }
        }
        // moderation process
        InstitutionalPageEntity institutionalPageEntity = institutionalPageModerationHandler
                .update(userId, institutionalPageId, retrievedInstitutionalPageEntity);
        // update institutional page
        entityBuilder.updateEntity(institutionalPageEntity, inputUpdateInstitutionalPageDto);
        // if page is being edited by someone else, then throws error
        UUID updateLockedBy = institutionalPageEntity.getUpdateLockedBy();
        if (updateLockedBy == null || updateLockedBy.equals(userId)) {
            institutionalPageEntity.setUpdateLockedBy(null);
        } else {
            throw new UpdateLockedException(institutionalPageId, updateLockedBy);
        }
        // save institutional page and return output dto
        InstitutionalPageEntity savedInstitutionalPageEntity = institutionalPageHelper.save(userId, institutionalPageEntity);
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPageEntity);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto deleteInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    ) {
        OutputInstitutionalPageDto outputInstitutionalPageDto;
        if (memberHelper.hasIPModerateRole()) {
            // retrieve institutional page entity
            InstitutionalPageEntity retrievedInstitutionalPageEntity = institutionalPageHelper
                    .retrieveLatestById(institutionalPageId);
            // verify if child institutional pages exist
            verifyNoChildrenOrThrow(institutionalPageId, retrievedInstitutionalPageEntity);
            // moderation process
            InstitutionalPageEntity institutionalPageEntity = institutionalPageModerationHandler
                    .prepareForImmediateDeletion(retrievedInstitutionalPageEntity);
            // build output dto, before deletion, otherwise we cannot set associations
            outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, institutionalPageEntity);
            // delete institutional page
            archiveHelper.persistArchivedCopy(institutionalPageEntity);
            institutionalPageHelper.delete(institutionalPageEntity);
        } else {
            // 'retrieveLatestForMemberById' already checks membership, but we call 'verifyWpLeaderOrThrow' first
            // to return meaningful errors: if 'institutionalPageId' refers to a published page,
            // it's clearer to return "not a member" instead of "institutional page not found"
            memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
            // retrieve institutional page entity
            InstitutionalPageEntity retrievedInstitutionalPageEntity = institutionalPageHelper
                    .retrieveLatestForMemberById(userId, institutionalPageId);
            // verify if child institutional pages exist
            verifyNoChildrenOrThrow(institutionalPageId, retrievedInstitutionalPageEntity);
            // moderation process
            InstitutionalPageEntity institutionalPageEntity = institutionalPageModerationHandler
                    .update(userId, institutionalPageId, retrievedInstitutionalPageEntity);
            // update institutional page
            institutionalPageEntity.setToDelete(true);
            // if page is being edited by someone else, then throws error
            UUID updateLockedBy = institutionalPageEntity.getUpdateLockedBy();
            if (updateLockedBy == null || updateLockedBy.equals(userId)) {
                institutionalPageEntity.setUpdateLockedBy(null);
            } else {
                throw new UpdateLockedException(institutionalPageId, updateLockedBy);
            }
            // save institutional page
            InstitutionalPageEntity updatedInstitutionalPageEntity = institutionalPageHelper
                    .save(userId, institutionalPageEntity);
            // build output dto
            outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, updatedInstitutionalPageEntity);
        }
        // output dto
        return outputInstitutionalPageDto;
    }

    private void verifyNoChildrenOrThrow(UUID institutionalPageId, InstitutionalPageEntity retrievedInstitutionalPageEntity) {
        // We need to retrieve the original version of child, e.g. all the approved and also the pending
        // institutional pages which have been just created. This way we avoid to delete institutional page
        // where there are child that have just been created and waiting for approval.
        Set<UUID> childInstitutionalPageIds = institutionalPageHelper
                .retrieveOriginalChildInstitutionalPages(retrievedInstitutionalPageEntity)
                .stream()
                .map(AbstractUUIDEntity::getId)
                .collect(Collectors.toSet());
        if (!childInstitutionalPageIds.isEmpty()) {
            throw new InstitutionalPageHasChildrenException(institutionalPageId, childInstitutionalPageIds);
        }
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputInstitutionalPageDto> findInstitutionalPagesHierarchyByRootId(
            @NonNull UUID userId,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable,
            @NonNull UUID rootInstitutionalPageId
    ) {
        Page<InstitutionalPageEntity> institutionalPageEntityPage = institutionalPageHelper
                .retrieveAllApprovedHierarchyForMemberByRootId(userId, pageable, rootInstitutionalPageId);
        return dtoBuilder.mapEntitiesToDtos(userId, institutionalPageEntityPage, associationsToLoad);
    }


    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto updateFolderId(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull UUID newFolderId
    ) {
        // retrieve latest institutional page entity and update folder id
        InstitutionalPageEntity latestInstitutionalPageEntity = institutionalPageHelper
                .retrieveLatestById(institutionalPageId);
        latestInstitutionalPageEntity.setWorkspaceFolderId(newFolderId);
        // save latest institutional pages
        InstitutionalPageEntity savedInstitutionalPageEntity = institutionalPageHelper.save(userId, latestInstitutionalPageEntity);
        if (latestInstitutionalPageEntity.getOriginalInstitutionalPageId() != null) {
            // retrieve original institutional page entity and update folder id
            InstitutionalPageEntity originalInstitutionalPageEntity = institutionalPageHelper
                    .retrieveOriginalInstitutionalPage(latestInstitutionalPageEntity);
            originalInstitutionalPageEntity.setWorkspaceFolderId(newFolderId);
            // save original institutional pages
            savedInstitutionalPageEntity = institutionalPageHelper.save(userId, originalInstitutionalPageEntity);
        }
        // skip the moderation process
        // return output dto
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPageEntity);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputInstitutionalPageDto> findAllPublishedInstitutionalPages(
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable
    ) {
        Page<InstitutionalPageEntity> institutionalPageEntityPage = institutionalPageHelper
                .retrieveAllPublished(pageable);
        return dtoBuilder.mapEntitiesToDtos(null, institutionalPageEntityPage, associationsToLoad);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputInstitutionalPageDto> searchPublishedInstitutionalPages(
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable,
            @NonNull InputSearchPublishedInstitutionalPageDto inputSearchPublishedInstitutionalPageDto
    ) {
        SearchInstitutionalPageModel searchInstitutionalPageModel = dtoBuilder
                .toFilters(inputSearchPublishedInstitutionalPageDto);
        Page<InstitutionalPageEntity> institutionalPageEntityPage = institutionalPageHelper
                .retrieveAllApprovedWithFilter(pageable, searchInstitutionalPageModel);
        return dtoBuilder.mapEntitiesToDtos(null, institutionalPageEntityPage, associationsToLoad);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputInstitutionalPageDto findPublishedInstitutionalPageById(
            @NonNull UUID institutionalPageId
    ) {
        InstitutionalPageEntity institutionalPageEntity = institutionalPageHelper
                .retrievePublishedById(institutionalPageId);
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(null, institutionalPageEntity);
        return outputInstitutionalPageDto;
    }


    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputInstitutionalPageDto> findAllModerationInstitutionalPages(
            @NonNull UUID userId,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable
    ) {
        Page<InstitutionalPageEntity> institutionalPageEntityPage = institutionalPageHelper.retrieveAllPendingOrRejected(pageable);
        return dtoBuilder.mapEntitiesToDtos(userId, institutionalPageEntityPage, associationsToLoad);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputInstitutionalPageDto> searchModerationInstitutionalPages(
            @NonNull UUID userId,
            @NonNull Set<String> associationsToLoad,
            @NonNull Pageable pageable,
            @NonNull InputSearchModerationInstitutionalPageDto inputSearchModerationInstitutionalPageDto
    ) {
        SearchInstitutionalPageModel searchInstitutionalPageModel = dtoBuilder.toFilters(inputSearchModerationInstitutionalPageDto);
        Page<InstitutionalPageEntity> institutionalPageEntityPage = institutionalPageHelper.retrieveAllPendingOrRejectedWithFilter(pageable, searchInstitutionalPageModel, inputSearchModerationInstitutionalPageDto.getModerationStatus());
        return dtoBuilder.mapEntitiesToDtos(userId, institutionalPageEntityPage, associationsToLoad);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputInstitutionalPageDto findModerationInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    ) {
        InstitutionalPageEntity institutionalPageEntity = institutionalPageHelper.retrievePendingOrRejectedById(institutionalPageId);
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, institutionalPageEntity);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputModerationDto acceptOrRejectPendingInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull InputPatchIPModerationRequestDto inputPatchIPModerationRequestDto
    ) {
        // verify user is admin
        memberHelper.verifyIPModeratorOrThrow();
        // retrieve most updated institutional page
        InstitutionalPageEntity institutionalPageEntity = institutionalPageHelper
                .retrieveLatestById(institutionalPageId);
        ModerationOperationType operationType = getModerationOperationType(institutionalPageEntity);
        InstitutionalPageEntity savedInstitutionalPageEntity;
        if (inputPatchIPModerationRequestDto.getApproved()) {
            institutionalPageEntity = institutionalPageModerationHandler.approve(
                    institutionalPageId,
                    institutionalPageEntity
            );
            if (institutionalPageEntity.isToDelete()) {
                archiveHelper.persistArchivedCopy(institutionalPageEntity);
                institutionalPageHelper.delete(institutionalPageEntity);
                savedInstitutionalPageEntity = institutionalPageEntity;
            } else {
                savedInstitutionalPageEntity = institutionalPageHelper.save(userId, institutionalPageEntity);
            }
        } else {
            institutionalPageEntity = institutionalPageModerationHandler.reject(
                    institutionalPageId,
                    institutionalPageEntity,
                    inputPatchIPModerationRequestDto.getMessage()
            );
            savedInstitutionalPageEntity = institutionalPageHelper.save(userId, institutionalPageEntity);
        }
        // return output dto
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPageEntity);
        return OutputModerationDto.builder()
                .operationType(operationType.getLabel())
                .institutionalPage(outputInstitutionalPageDto)
                .build();
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto requestModerationInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    ) {
        // 'retrieveLatestForMemberById' already checks membership, but we call 'verifyWpLeaderOrThrow' first
        // to return meaningful errors: if 'institutionalPageId' refers to a published page,
        // it's clearer to return "not a member" instead of "institutional page not found"
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page entity
        InstitutionalPageEntity retrievedInstitutionalPageEntity = institutionalPageHelper
                .retrieveLatestForMemberById(userId, institutionalPageId);
        // moderation process
        InstitutionalPageEntity institutionalPageEntity = institutionalPageModerationHandler
                .requestModeration(retrievedInstitutionalPageEntity);
        // save institutional page and return output dto
        InstitutionalPageEntity savedInstitutionalPageEntity = institutionalPageHelper.save(userId, institutionalPageEntity);
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPageEntity);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto requestPublicationInstitutionalPage(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    ) {
        // 'retrieveLatestForMemberById' already checks membership, but we call 'verifyWpLeaderOrThrow' first
        // to return meaningful errors: if 'institutionalPageId' refers to a published page,
        // it's clearer to return "not a member" instead of "institutional page not found"
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page entity
        InstitutionalPageEntity retrievedInstitutionalPageEntity = institutionalPageHelper
                .retrieveLatestForMemberById(userId, institutionalPageId);
        // moderation process
        InstitutionalPageEntity institutionalPageEntity = institutionalPageModerationHandler
                .update(userId, institutionalPageId, retrievedInstitutionalPageEntity);
        // update institutional page
        institutionalPageEntity.setPublished(true);
        // save institutional page and return output dto
        InstitutionalPageEntity savedInstitutionalPageEntity = institutionalPageHelper.save(userId, institutionalPageEntity);
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPageEntity);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputRequestUpdateDto requestUpdateInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    ) {
        // 'retrieveLatestForMemberById' already checks membership, but we call 'verifyWpLeaderOrThrow' first
        // to return meaningful errors: if 'institutionalPageId' refers to a published page,
        // it's clearer to return "not a member" instead of "institutional page not found"
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page entity
        InstitutionalPageEntity institutionalPageEntity = institutionalPageHelper
                .retrieveLatestForMemberById(userId, institutionalPageId);
        // if page is in PENDING state, then throw error
        if (institutionalPageEntity.getModerationStatus().equals(ModerationStatus.PENDING)) {
            throw new InvalidModerationStatusException(institutionalPageId, institutionalPageEntity.getModerationStatus());
        }
        // if not locked by someone else, then lock the institutional page
        if (institutionalPageEntity.getUpdateLockedBy() == null) {
            // set update locked by and save institutional page and
            institutionalPageEntity.setUpdateLockedBy(userId);
            institutionalPageEntity = institutionalPageHelper.save(userId, institutionalPageEntity);
        }
        // if already locked by current user, 'updateLockObtained' field will be true
        boolean updateLockObtained = institutionalPageEntity.getUpdateLockedBy().equals(userId);
        // return output dto
        return OutputRequestUpdateDto
                .builder()
                .updateLockObtained(updateLockObtained)
                .updateLockedBy(institutionalPageEntity.getUpdateLockedBy())
                .build();
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void cancelUpdateInstitutionalPageById(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId
    ) {
        // 'retrieveLatestForMemberById' already checks membership, but we call 'verifyWpLeaderOrThrow' first
        // to return meaningful errors: if 'institutionalPageId' refers to a published page,
        // it's clearer to return "not a member" instead of "institutional page not found"
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page entity
        InstitutionalPageEntity institutionalPageEntity = institutionalPageHelper
                .retrieveLatestForMemberById(userId, institutionalPageId);
        // if locked by current user, then release lock
        if (userId.equals(institutionalPageEntity.getUpdateLockedBy())) {
            // set update locked by and save institutional page and
            institutionalPageEntity.setUpdateLockedBy(null);
            institutionalPageHelper.save(userId, institutionalPageEntity);
        } else {
            throw new UpdateLockedException(institutionalPageId, institutionalPageEntity.getUpdateLockedBy());
        }
    }

    // private

    private static ModerationOperationType getModerationOperationType(InstitutionalPageEntity institutionalPageEntity) {
        ModerationOperationType operationType;
        if (institutionalPageEntity.isToDelete()) {
            operationType = ModerationOperationType.DELETE;
        } else if (institutionalPageEntity.getOriginalInstitutionalPageId() == null) {
            operationType = ModerationOperationType.CREATE;
        } else {
            operationType = ModerationOperationType.UPDATE;
        }
        return operationType;
    }

}
