package com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.PrePersist;
import jakarta.persistence.Table;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.UUID;

/**
 * This entity represents a soft deleted {@link MemberEntity}.
 */
@Entity
@Table(name = "archived_institutional_page_member")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedMemberEntity extends AbstractUUIDEntity {

    /**
     * The memberId.
     */
    @Column(name = "member_id", nullable = false)
    private UUID memberId;

    /**
     * Association of the related institutional page for navigation.
     */
    @ManyToOne(fetch = FetchType.LAZY, optional = false)
    @JoinColumn(name = "institutional_page_id", updatable = false)
    private ArchivedInstitutionalPageEntity institutionalPage;

    /**
     * ID of the related institutional page.
     * <p></p>
     * Had to add this column to avoid joins for findByInstitutionalPageId* queries.
     * <a href="https://github.com/spring-projects/spring-data-jpa/issues/1575">JPA Github Issue</a>
     */
    @Column(name = "institutional_page_id", insertable = false, updatable = false)
    private UUID institutionalPageId;

    /**
     * Indicates whether the user is a WP lead.
     */
    @Column(name = "wp_lead", nullable = false, updatable = false)
    private boolean wpLead;

    /**
     * The timestamp when the institutionalPage was created
     * in the original (non-archived) table.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the institutionalPage was last updated
     * in the original (non-archived) table.
     */
    @Column(name = "update_time", nullable = false, updatable = false)
    private Instant updateTime;

    /**
     * The timestamp when this record was inserted into the archive table,
     * representing the moment the institutionalPage was archived.
     */
    @Column(name = "archival_time", nullable = false, updatable = false)
    private Instant archivalTime;

    /**
     * Sets {@code archivalTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PrePersist
    public void onCreate() {
        archivalTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
    }

}
