package com.finconsgroup.itserr.marketplace.metadata.bs.messaging.impl;


import com.finconsgroup.itserr.marketplace.metadata.bs.config.MetadataBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.metadata.bs.dto.OutputMetadataDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.mapper.MetadataMapper;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.dto.MessageBodyDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.dto.MetadataCreatedMessageBodyDto;
import com.finconsgroup.itserr.marketplace.metadata.bs.messaging.dto.MetadataDeletedMessageBodyDto;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.UUID;

/**
 * Default implementation of {@link ResourceProducer} that publishes
 * resource events (create, update, delete) for catalog items.
 */
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Slf4j
public class DefaultResourceProducer implements ResourceProducer {

    /**
     * Metadata mapper.
     */
    private final MetadataMapper metadataMapper;

    private final MessageProducer messageProducer;

    private final MetadataBsConfigurationProperties.Messaging messagingProperties;

    /**
     * Constructs the DefaultResourceProducer.
     *
     * @param metadataBsConfigurationProperties application-specific configuration properties
     * @param producerRegistry                  the registry providing message producers
     */
    public DefaultResourceProducer(
            MetadataBsConfigurationProperties metadataBsConfigurationProperties,
            MetadataMapper metadataMapper,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = metadataBsConfigurationProperties.getMessaging();
        this.metadataMapper = metadataMapper;
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getProducerName());
    }

    @Override
    public void publishCreatedResource(@NonNull OutputMetadataDto metadata) {
        // convert to message body DTO
        final MetadataCreatedMessageBodyDto resource = metadataMapper.dtoToCreatedMessageDto(metadata);
        publish(resource, messagingProperties.getResourceCreatedEventType());
    }

    @Override
    public void publishDeletedResource(@NonNull UUID documentId) {
        // convert to message body DTO
        final MetadataDeletedMessageBodyDto resource = MetadataDeletedMessageBodyDto.builder().id(documentId).build();
        publish(resource, messagingProperties.getResourceDeletedEventType());
    }

    /**
     * Sends a message to the messaging system.
     *
     * @param resource  the payload of the message
     * @param eventType the type of event to publish
     * @throws WP2MessagingException if message publishing fails
     */
    private void publish(@NonNull MessageBodyDto resource, @NonNull String eventType) {
        try {
            Map<String, MessageBodyDto> messageMap = Map.of(String.valueOf(resource.getId()), resource);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending message, messageMap: %s, eventType: %s, source: %s, filterProperties: %s"
                    .formatted(messageMap, eventType, messagingProperties.getSource(), filterProperties));
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published message for resource with id: %s".formatted(resource.getId()));
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage);
        }
    }
}
