package com.finconsgroup.itserr.marketplace.news.bs.api;

import com.finconsgroup.itserr.marketplace.news.bs.enums.EventType;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.core.io.Resource;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestParam;

/**
 * This interface defines the contract for REST API endpoints used for system testing and debugging.
 *
 * <p>
 * These endpoints are intended for internal use only and
 * provide endpoints used for debugging and runtime analysis.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET /api/v1/bs/diagnostics/download-logs - Download application log for analysis
 * </pre>
 * </p>
 */
@Tag(
        name = "Diagnostics",
        description = "Diagnostics API: provides endpoints used for " +
                "debugging and runtime analysis."
)
@SecurityRequirement(name = "BearerAuth")
public interface DiagnosticsApi {

    @Operation(
            summary = "Download log file",
            responses = {@ApiResponse(responseCode = "200", description = "OK"),}
    )
    @GetMapping(path = "/news/diagnostics/download-logs", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    ResponseEntity<Resource> downloadLogs();

    /**
     * Test API method for sending a message about a news event.
     * Used to simulate or trigger the messaging logic for item creation, update, or deletion.
     *
     * @param eventType the type of event (CREATED, UPDATED or DELETED)
     * @param name      the name of the news (defaults to "item name" if not provided)
     * @return ResponseEntity with operation result
     * @throws IllegalArgumentException if the event type is invalid
     */
    @Operation(
            summary = "Test method to send news event message",
            description = "Simulates publishing events for news changes (creation, update or deletion)",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Message sent successfully"),
                    @ApiResponse(responseCode = "400", description = "Invalid request parameters"),
                    @ApiResponse(responseCode = "404", description = "Item not found (for update/delete operations)")
            }
    )
    @GetMapping(path = "/news/diagnostics/publish-message", produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<String> publishMessage(
            @RequestParam(name = "eventType", required = false) EventType eventType,
            @RequestParam(name = "name", required = false, defaultValue = "item name") String name
    );
}
