package com.finconsgroup.itserr.marketplace.news.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.OutputImageDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.constraints.NotBlank;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.multipart.MultipartFile;

/**
 * This interface defines the contract for REST API endpoints related to Image Management.
 *
 * <p>
 * It provides endpoints for creating/updating images.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * POST /images - Create a new image.
 * </pre>
 * </p>
 */
@Tag(name = "Images", description = "Operations related to images")
@SecurityRequirement(name = "BearerAuth")
public interface ImageApi {

    /**
     * Creates an image file, creating a new version of the document.
     *
     * @param file The new image file.
     * @return Created image.
     */
    @Operation(
            summary = "Creates a new image file.",
            description = "Returns the created image document"
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Image document created")
    })
    @PostMapping(path = "/news/images", consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.CREATED)
    OutputImageDto createPublicImage(@RequestPart MultipartFile file);

    /**
     * Replaces an image file, creating a new version of the document.
     *
     * @param id   id of the image to be updated.
     * @param file The new image file.
     * @return Updated image.
     */
    @Operation(
            summary = "Replaces an image file, creating a new version of the document.",
            description = "Returns the updated image document"
    )
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Image document updated"),
            @ApiResponse(responseCode = "404", description = "Document not found", content = @Content(
                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ErrorResponseDto.class)
            )),
            @ApiResponse(responseCode = "422", description = "Id does not match a document", content = @Content(
                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                    schema = @Schema(implementation = ErrorResponseDto.class)
            )),
    })
    @PutMapping(path = "/news/images/{id}", consumes = MediaType.MULTIPART_FORM_DATA_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputImageDto updatePublicImage(@PathVariable @NotBlank String id, @RequestPart MultipartFile file);

}