package com.finconsgroup.itserr.marketplace.news.bs.client;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.InputCreateNewsDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.InputUpdateNewsDto;
import com.finconsgroup.itserr.marketplace.news.bs.client.dto.newsDm.OutputNewsDto;
import jakarta.validation.Valid;
import org.springframework.data.domain.Sort;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.UUID;

/**
 * Feign client for communication with the News DM service.
 */
public interface NewsDmClient {

    /**
     * Retrieves a paginated list of all news.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "startTime")
     * @param direction  the direction of sorting (default is descending)
     * @return a page of {@link OutputPageDto<OutputNewsDto>} and HTTP status 200 (OK)
     */
    @GetMapping(value = "/news/news", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<OutputNewsDto> findAll(
            @RequestParam(name = "pageNumber") int pageNumber,
            @RequestParam(name = "pageSize") int pageSize,
            @RequestParam(name = "sort") String sort,
            @RequestParam(name = "direction") Sort.Direction direction
    );

    /**
     * Retrieves news by id.
     *
     * @param id the id of the news to retrieve
     * @return the found {@link OutputNewsDto} and HTTP status 200 (OK)
     */
    @GetMapping(value = "/news/news/{id}")
    OutputNewsDto findById(@PathVariable("id") UUID id);

    /**
     * Creates new news.
     *
     * @param inputCreateNewsDto the input data transfer object containing news details
     */
    @PostMapping(value = "/news/news")
    OutputNewsDto createNews(@Valid @RequestBody InputCreateNewsDto inputCreateNewsDto);

    /**
     * Updates existing news by id.
     *
     * @param id                 the id of the news to update
     * @param inputUpdateNewsDto the input data transfer object containing updated news details
     * @return the updated {@link OutputNewsDto} and HTTP status 200 (OK)
     */
    @PutMapping(value = "/news/news/{id}")
    OutputNewsDto updateNewsById(
            @PathVariable("id") UUID id,
            @Valid @RequestBody InputUpdateNewsDto inputUpdateNewsDto
    );

    /**
     * Deletes news by id.
     *
     * @param id the id of the news to delete
     */
    @DeleteMapping(value = "/news/news/{id}")
    void deleteNewsById(@PathVariable("id") UUID id);
}
