package com.finconsgroup.itserr.marketplace.news.bs.component;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.news.bs.bean.NewsApplicationEvent;
import com.finconsgroup.itserr.marketplace.news.bs.dto.OutputNewsDetailDto;
import com.finconsgroup.itserr.marketplace.news.bs.enums.EventType;
import com.finconsgroup.itserr.marketplace.news.bs.mapper.NewsMapper;
import com.finconsgroup.itserr.marketplace.news.bs.messaging.EventProducer;
import com.finconsgroup.itserr.marketplace.news.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.marketplace.news.bs.messaging.dto.NewsCreatedMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.news.bs.messaging.dto.NewsMessagingAdditionalDataDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

/**
 * The implementation for {@link ApplicationListener} to listen to News application events
 * and produce relevant messages.
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class MessagingNewsApplicationEventListener implements ApplicationListener<NewsApplicationEvent> {

    private final EventProducer eventProducer;
    private final ResourceProducer resourceProducer;
    private final NewsMapper newsMapper;

    @Override
    public void onApplicationEvent(@NonNull NewsApplicationEvent event) {

        MessagingEventDto<NewsMessagingAdditionalDataDto> messagingEventDto =
                newsMapper.toMessagingEventDto(event.getNews());

        MessagingEventUserDto user = MessagingEventUserDto.builder()
                .id(JwtTokenHolder.getUserId().orElse(null))
                .name(JwtTokenHolder.getName().orElse(null))
                .username(JwtTokenHolder.getPreferredUsername().orElse(null))
                .build();
        messagingEventDto.setUser(user);

        updateEventForType(event, messagingEventDto);
        publishEventForType(event, messagingEventDto);
    }

    private void updateEventForType(NewsApplicationEvent applicationEvent, MessagingEventDto<NewsMessagingAdditionalDataDto> messagingEventDto) {
        OutputNewsDetailDto outputNewsDto = applicationEvent.getNews();
        NewsMessagingAdditionalDataDto additionalData = createAdditionalData(applicationEvent.getEventType());

        switch (applicationEvent.getEventType()) {
            case CREATED -> {
                messagingEventDto.setTimestamp(outputNewsDto.getCreationTime());
                updateAdditionalDataForCreated(additionalData, outputNewsDto);
            }
            case UPDATED, DELETED -> messagingEventDto.setTimestamp(applicationEvent.getEventTimestamp());

        }

        messagingEventDto.setAdditionalData(additionalData);
    }

    private NewsMessagingAdditionalDataDto createAdditionalData(EventType eventType) {
        if (eventType == EventType.CREATED) {
            return new NewsCreatedMessagingAdditionalDataDto();
        } else {
            return new NewsMessagingAdditionalDataDto();
        }
    }

    private void updateAdditionalDataForCreated(NewsMessagingAdditionalDataDto additionalData,
                                                OutputNewsDetailDto outputNewsDto) {
        if (additionalData instanceof NewsCreatedMessagingAdditionalDataDto createdAdditionalDataDto) {
            createdAdditionalDataDto.setContent(outputNewsDto.getContent());
            if (outputNewsDto.getImage() != null && outputNewsDto.getImage().getUrl() != null) {
                createdAdditionalDataDto.setImageUrl(outputNewsDto.getImage().getUrl().toString());
            }
        }
    }

    private void publishEventForType(NewsApplicationEvent applicationEvent, MessagingEventDto<?> messagingEventDto) {
        switch (applicationEvent.getEventType()) {
            case CREATED -> {
                resourceProducer.publishCreatedResource(applicationEvent.getNews());
                eventProducer.publishCreateEvent(messagingEventDto);
            }
            case UPDATED -> {
                resourceProducer.publishUpdatedResource(applicationEvent.getNews());
                eventProducer.publishUpdateEvent(messagingEventDto);
            }
            case DELETED -> {
                // publish to delete resource from global search
                resourceProducer.publishDeletedResource(applicationEvent.getNews());
                eventProducer.publishDeleteEvent(messagingEventDto);
            }
        }
    }

}
