package com.finconsgroup.itserr.marketplace.news.bs.messaging.impl;


import com.finconsgroup.itserr.marketplace.news.bs.config.properties.NewsBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.news.bs.messaging.EventProducer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;

import static com.finconsgroup.itserr.marketplace.news.bs.util.Constants.SERVICE_FILTER_KEY;
import static com.finconsgroup.itserr.marketplace.news.bs.util.Constants.SERVICE_FILTER_VALUE;


/**
 * Default implementation of {@link EventProducer} that publishes
 * resource events (create, update, delete) for news.
 */
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Slf4j
public class DefaultEventProducer implements EventProducer {

    private final MessageProducer messageProducer;

    private final NewsBsConfigurationProperties.Messaging messagingProperties;

    /**
     * Constructs the DefaultEventProducer.
     *
     * @param newsBsConfigurationProperties application-specific configuration properties
     * @param producerRegistry              the registry providing message producers
     */
    public DefaultEventProducer(
            NewsBsConfigurationProperties newsBsConfigurationProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = newsBsConfigurationProperties.getMessaging();
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getEventProducerName());
    }

    @Override
    public void publishCreateEvent(@NotNull MessagingEventDto<?> resource) {
        publish(resource, messagingProperties.getResourceCreatedEventType());
    }

    @Override
    public void publishUpdateEvent(@NotNull MessagingEventDto<?> resource) {
        publish(resource, messagingProperties.getResourceUpdatedEventType());
    }

    @Override
    public void publishDeleteEvent(@NotNull MessagingEventDto<?> resource) {
        publish(resource, messagingProperties.getResourceDeletedEventType());
    }

    /**
     * Sends a message to the messaging system.
     *
     * @param eventDto  the payload of the message
     * @param eventType the type of event to publish
     * @throws WP2MessagingException if message publishing fails
     */
    private void publish(@NonNull MessagingEventDto<?> eventDto, @NonNull String eventType) {
        try {
            Map<String, MessagingEventDto<?>> messageMap = Map.of(eventDto.getId(), eventDto);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending message, messageMap: {}, eventType: {}, source: {}, filterProperties: {}",
                    messageMap, eventType, messagingProperties.getSource(), filterProperties);
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published event message for resource with id: {}", eventDto.getId());
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending event message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage);
        }
    }
}
