package com.finconsgroup.itserr.marketplace.notification.bs.websocket;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.notification.bs.client.NotificationInternalDmClient;
import com.finconsgroup.itserr.marketplace.notification.bs.client.dto.OutputLocalizedUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.bs.event.NotificationCreatedEvent;
import com.finconsgroup.itserr.marketplace.notification.bs.mapper.NotificationMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;
import org.springframework.lang.NonNull;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

/**
 * Component responsible for publishing notifications to websocket connections. Listens for notification events and broadcasts them to appropriate websocket
 * sessions.
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class NotificationWebsocketPublisher {

    /** Client for interacting with the Notification DM microservice. */
    private final NotificationInternalDmClient notificationInternalDmClient;

    /** Handler for managing websocket connections and message distribution */
    private final NotificationsWebSocketHandler notificationsWebSocketHandler;

    /** Mapper responsible for transforming notification data. */
    private final NotificationMapper notificationMapper;

    /** Object mapper for serializing notifications to JSON format */
    private final ObjectMapper objectMapper;

    /**
     * Handles notification created events by publishing them to websocket connections. Serializes the notification to JSON and sends it to the appropriate
     * user's websocket session.
     *
     * @param event the notification created event to handle
     */
    @Async
    @EventListener
    public void onNotificationCreated(@NonNull final NotificationCreatedEvent event) {

        log.debug("Publishing notification {} to websockets", event.getId());
        notificationsWebSocketHandler.sendNoThrow(
                event.getUser(),
                session -> buildMessage(event, session));

    }

    private String buildMessage(
            @NonNull final NotificationCreatedEvent event,
            @NonNull final WebSocketSessionInfo session) {

        final OutputLocalizedUserNotificationDto notification = notificationInternalDmClient.findById(
                event.getId(),
                session.acceptLanguage());

        try {
            return objectMapper.writeValueAsString(
                    notificationMapper.toWsDto(notification));
        } catch (Exception e) {
            throw new IllegalStateException("Error serializing notification", e);
        }

    }

}
