package com.finconsgroup.itserr.marketplace.notification.bs.websocket;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.CloseStatus;
import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;

/**
 * WebSocket handler for managing notification connections and message distribution. Handles connection lifecycle and message broadcasting to connected
 * clients.
 */
@Slf4j
@Component
public class NotificationsWebSocketHandler extends TextWebSocketHandler {

    /** Set of active WebSocket sessions maintained by this handler */
    private final Set<WebSocketSession> sessions = new HashSet<>();

    /**
     * Handles new WebSocket connection establishment. Adds the session to active sessions.
     *
     * @param session the WebSocket session that was established
     */
    @Override
    public void afterConnectionEstablished(@NonNull final WebSocketSession session) {

        // Add session
        final int sessionsCount;
        synchronized (sessions) {
            sessions.add(session);
            sessionsCount = sessions.size();
        }

        // Log
        final String username = getUsername(session);
        log.debug("Websocket connected: user={}, sessions={}", username, sessionsCount);

    }

    @Nullable
    private static String getUsername(@NonNull final WebSocketSession session) {
        return Optional.of(session)
                .map(WebSocketSession::getAttributes)
                .map(m -> (String) m.get(Attributes.USERNAME_ATTR))
                .orElse(null);
    }

    @Nullable
    private static UUID getUserId(@NonNull final WebSocketSession session) {
        return Optional.of(session)
                .map(WebSocketSession::getAttributes)
                .map(m -> (UUID) m.get(Attributes.USER_ID_ATTR))
                .orElse(null);
    }

    @Nullable
    private static String getUserIdAsString(@NonNull final WebSocketSession session) {
        return Optional.ofNullable(getUserId(session))
                .map(UUID::toString)
                .orElse(null);
    }

    @Nullable
    private static String getEmail(@NonNull final WebSocketSession session) {
        return Optional.of(session)
                .map(WebSocketSession::getAttributes)
                .map(m -> (String) m.get(Attributes.EMAIL_ATTR))
                .orElse(null);
    }

    @Nullable
    private static String getAcceptLanguage(@NonNull final WebSocketSession session) {
        return Optional.of(session)
                .map(WebSocketSession::getAttributes)
                .map(m -> (String) m.get(Attributes.ACCEPT_LANGUAGE_ATTR))
                .orElse(null);
    }

    @Override
    protected void handleTextMessage(
            @NonNull final WebSocketSession session,
            @NonNull final TextMessage message) {
        // Nothing to handle
    }

    /**
     * Handles WebSocket connection closure. Removes the session from active sessions.
     *
     * @param session the WebSocket session that was closed
     * @param status the status indicating why the connection was closed
     */
    @Override
    public void afterConnectionClosed(
            @NonNull final WebSocketSession session,
            @NonNull final CloseStatus status) {
        synchronized (sessions) {
            sessions.remove(session);
        }
        log.debug(
                "Websocket closed: user={}, status={}, reason={}",
                getUsername(session),
                status.getCode(),
                status.getReason());
    }

    /**
     * Finds all active WebSocket sessions that are associated with a given user ID. Method is thread safe.
     *
     * @param user the user whose WebSocket sessions need to be retrieved, expressed as an id, a username or an email
     * @return a list of WebSocket sessions associated with the specified user ID, or an empty list if no sessions are found
     */
    @NonNull
    private List<WebSocketSession> findSessionsByUser(@NonNull final String user) {

        final String normalizedUser = StringUtils.lowerCase(user);
        synchronized (sessions) {
            return sessions.stream()
                    .filter(s ->
                            normalizedUser == null
                                    || normalizedUser.equals(StringUtils.lowerCase(getUserIdAsString(s)))
                                    || normalizedUser.equals(StringUtils.lowerCase(getUsername(s)))
                                    || normalizedUser.equals(StringUtils.lowerCase(getEmail(s))))
                    .toList();
        }

    }

    /**
     * Sends a message to all connected sessions without throwing exceptions.
     *
     * @param message the message to send
     */
    public void sendNoThrow(final String message) {
        sendNoThrow((String) null, message);
    }

    /**
     * Sends a message to sessions associated with a specific user without throwing exceptions. If user is null, sends to all connected sessions.
     *
     * @param user the target user ID, or null for all users
     * @param message the message to send
     */
    public void sendNoThrow(
            final String user,
            final String message) {

        // Find message target sessions
        final List<WebSocketSession> targetSessions;
        synchronized (sessions) {
            targetSessions = user != null
                    ? findSessionsByUser(user)
                    : new ArrayList<>(sessions);
        }

        // Send message
        for (final WebSocketSession session : targetSessions) {
            sendNoThrow(session, message);
        }

    }

    /**
     * Sends a message to WebSocket sessions associated with a specific user ID without throwing exceptions. If the user is null, the message is sent to all
     * active sessions. The message is built using the provided messageBuilder function, which takes session-specific information as input.
     *
     * @param user the target user, expressed as id, username or email; if null, the message is sent to all connected sessions
     * @param messageBuilder a function that builds the message to send based on the WebSocket session info
     */
    public void sendNoThrow(
            final String user,
            @NonNull final Function<WebSocketSessionInfo, String> messageBuilder) {

        // Find message target sessions
        final List<WebSocketSession> targetSessions;
        synchronized (sessions) {
            targetSessions = user != null
                    ? findSessionsByUser(user)
                    : new ArrayList<>(sessions);
        }

        // Send message
        for (final WebSocketSession session : targetSessions) {

            // Build session information
            final WebSocketSessionInfo info = new WebSocketSessionInfo(
                    getUserId(session),
                    getAcceptLanguage(session));

            // Building message
            final String message;
            try {
                message = messageBuilder.apply(info);
            } catch (final Exception e) {
                log.error("Error building websocket message", e);
                continue;
            }

            // Send message
            sendNoThrow(session, message);

        }

    }

    /**
     * Sends a message to the specified WebSocket session without throwing exceptions. If an error occurs during message sending, it is logged but no exception
     * is propagated.
     *
     * @param session the WebSocket session to which the message should be sent
     * @param message the message to send to the specified session
     */
    private void sendNoThrow(
            final WebSocketSession session,
            final String message) {

        try {
            send(session, message);
        } catch (Exception e) {
            log.error("Error sending websocket message", e);
        }

    }

    private void send(
            final WebSocketSession session,
            final String message) throws IOException {

        session.sendMessage(new TextMessage(message));

    }

}
