package com.finconsgroup.itserr.marketplace.notification.bs.websocket;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import io.micrometer.common.util.StringUtils;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;
import org.springframework.web.socket.WebSocketHandler;
import org.springframework.web.socket.server.HandshakeInterceptor;

import java.util.Map;
import java.util.UUID;

/**
 * WebSocket handshake interceptor that validates user authentication and adds user attributes to the WebSocket session. Ensures that only authenticated users
 * can establish WebSocket connections.
 */
@Component
public class WsHandshakeInterceptor implements HandshakeInterceptor {

    /**
     * Intercepts WebSocket handshake before it's established to validate user authentication and add user attributes to the session.
     *
     * @param request the HTTP request for the WebSocket handshake
     * @param response the HTTP response for the WebSocket handshake
     * @param wsHandler the WebSocket handler that will process the connection
     * @param attributes map for storing session attributes
     * @return true if handshake should proceed, false to reject it
     * @throws Exception if an error occurs during handshake
     */
    @Override
    public boolean beforeHandshake(
            @NonNull final ServerHttpRequest request,
            @NonNull final ServerHttpResponse response,
            @NonNull final WebSocketHandler wsHandler,
            @NonNull final Map<String, Object> attributes) throws Exception {

        final UUID userId = JwtTokenHolder.getUserId().orElse(null);
        final String username = JwtTokenHolder.getPreferredUsername().orElse(null);
        final String email = JwtTokenHolder.getEmail().orElse(null);
        final String acceptLanguage = request.getHeaders().getFirst("Accept-Language");
        if (userId == null || StringUtils.isBlank(username) || StringUtils.isBlank(email)) {
            return false;
        }

        attributes.put(Attributes.USER_ID_ATTR, userId);
        attributes.put(Attributes.USERNAME_ATTR, username);
        attributes.put(Attributes.EMAIL_ATTR, email);
        if (StringUtils.isNotBlank(acceptLanguage)) {
            attributes.put(Attributes.ACCEPT_LANGUAGE_ATTR, acceptLanguage);
        }
        return true;
    }

    @Override
    public void afterHandshake(
            @NonNull final ServerHttpRequest request,
            @NonNull final ServerHttpResponse response,
            @NonNull final WebSocketHandler wsHandler,
            final Exception exception) {
        // Do nothing
    }
}
