package com.finconsgroup.itserr.marketplace.notification.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;
import org.hibernate.annotations.JdbcTypeCode;
import org.hibernate.type.SqlTypes;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Map;

/**
 * Entity class representing a user notification in the notification system. This entity is mapped to the "user_notification" table in the database and stores
 * information such as the user ID, notification type, referenced entity, and status flags.
 *
 * <p>Example usage:
 * <pre>
 * UserNotificationEntity notification = UserNotificationEntity.builder()
 *     .userId(UUID.randomUUID())
 *     .notificationType("catalog-item-created")
 *     .referencedId("12345")
 *     .placeholderValues(Map.of("userName", "John Doe"))
 *     .build();
 * </pre>
 * </p>
 */
@Entity
@Table(name = "user_notification")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class UserNotificationEntity extends AbstractUUIDEntity {

    /**
     * <p>The user receiving the notification.</p>
     * <p>
     * May be:
     *     <ul>
     *         <li>the user UUID</li>
     *         <li>the user username</li>
     *         <li>the user e-mail</li>
     *     </ul>
     * </p>
     */
    @Column(name = "user_ref", nullable = false)
    private String user;

    /**
     * The type of notification (e.g., "catalog-item-event-created", "catalog-item-event-status").
     */
    @Column(name = "notification_type", nullable = false, length = 100)
    private String type;

    /**
     * The ID of the referenced entity that triggered this notification.
     */
    @Column(name = "referenced_id", nullable = false, length = 100)
    private String referencedId;

    /**
     * Placeholder values for dynamic notification content.
     */
    @Column(name = "placeholder_values", nullable = false, columnDefinition = "jsonb")
    @JdbcTypeCode(SqlTypes.JSON)
    private Map<String, String> placeholderValues;

    /**
     * Flag indicating whether the notification has been read.
     */
    @Column(name = "is_read")
    private Boolean read;

    /**
     * Flag indicating whether the notification has been archived.
     */
    @Column(name = "is_archived")
    private Boolean archived;

    /**
     * The timestamp when the notification was last read.
     */
    @Column(name = "last_read_time")
    private ZonedDateTime lastReadTime;

    /**
     * The timestamp when the notification was created.
     */
    @Column(name = "creation_time", nullable = false)
    private ZonedDateTime creationTime;

    /**
     * The timestamp when the notification was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private ZonedDateTime updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates. Each time the entity is updated, the version is incremented. If two transactions
     * try to update the same entity simultaneously, JPA will detect the conflict based on this version and throw an
     * {@link jakarta.persistence.OptimisticLockException}.
     */
    @Version
    @Column(name = "version", nullable = false)
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        ZonedDateTime now = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.MICROS);
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

}
