package com.finconsgroup.itserr.marketplace.notification.dm.service;

import com.finconsgroup.itserr.marketplace.notification.dm.dto.OutputLocalizedUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notification.dm.dto.UserDto;
import com.finconsgroup.itserr.marketplace.notification.dm.exception.UserNotificationNotFoundException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.UUID;

/**
 * Service for handling business logic related to UserNotification messages localization.
 */
public interface LocalizedUserNotificationService {

    /**
     * Retrieves a paginated list of all user notifications.
     *
     * @param user the user DTO containing identification information
     * @param read Filter notifications by read status. If null, returns all notifications regardless of read status.
     * @param archived Filter notifications by archived status. If null, returns all notifications regardless of archived status.
     * @param pageable the pagination information
     * @param httpAcceptLanguage HTTP Accept-Language header. May be null.
     * @return a page of OutputLocalizedUserNotificationDto representing the user notification
     */
    @NonNull
    Page<OutputLocalizedUserNotificationDto> findAll(
            @NonNull final UserDto user,
            final Boolean read,
            final Boolean archived,
            @NonNull Pageable pageable,
            final String httpAcceptLanguage);

    /**
     * Retrieves a user notification by notificationId (no security check).
     *
     * @param notificationId the contains the notificationId of the user notification to retrieve
     * @param httpAcceptLanguage HTTP Accept-Language header. May be null.
     * @return an OutputLocalizedUserNotificationDto representing the user notification details
     * @throws UserNotificationNotFoundException if no user notification is found for the given id
     */
    @NonNull
    OutputLocalizedUserNotificationDto findById(
            @NonNull UUID notificationId,
            final String httpAcceptLanguage);

    /**
     * Retrieves a user notification by notificationId.
     *
     * @param user the user DTO containing identification information
     * @param notificationId the notificationId of the user notification to retrieve
     * @param httpAcceptLanguage HTTP Accept-Language header. May be null.
     * @return an OutputLocalizedUserNotificationDto representing the user notification details
     * @throws UserNotificationNotFoundException if no user notification is found for the given id
     */
    @NonNull
    OutputLocalizedUserNotificationDto findByUser(
            @NonNull final UserDto user,
            @NonNull UUID notificationId,
            final String httpAcceptLanguage);

}
