package com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.client.NotificationBsClient;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.client.dto.InputCreateUserNotificationDto;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.config.properties.BusNotificationProperties;
import com.finconsgroup.itserr.marketplace.notificationfeeder.bs.event.extractor.FieldExtractorRegistry;
import io.cloudevents.CloudEvent;
import io.cloudevents.CloudEventData;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;

import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Handler for processing JSON-based bus notification messages received as CloudEvents. This class is responsible for handling a specific notification
 * configuration.
 */
@Slf4j
@RequiredArgsConstructor
public class JsonNotificationConsumerMessageHandler
        implements NotificationConsumerMessageHandler {

    /** The name identifier for this notification handler */
    private final String name;
    /** Configuration properties for bus notifications */
    private final BusNotificationProperties config;

    /** Instance of {@link ObjectMapper} used for JSON serialization and deserialization. */
    private final ObjectMapper objectMapper;

    /** Registry to extract fields values. */
    private final FieldExtractorRegistry fieldExtractorRegistry;

    /** Client used to communicate with the Notification BS microservice */
    private final NotificationBsClient notificationBsClient;

    @NonNull
    @Override
    public String getName() {
        return name;
    }

    @NonNull
    @Override
    public String getNotificationType() {
        return config.getType();
    }

    @Override
    public void handleMessage(final CloudEvent cloudEvent) throws Exception {

        // Read event raw data
        final CloudEventData rawData = cloudEvent.getData();
        if (rawData == null) {
            return;
        }

        // Get JSON string
        final String json = new String(rawData.toBytes());

        // Parse JSON
        JsonNode node;
        try {
            node = objectMapper.readTree(json);
        } catch (JsonProcessingException e) {
            log.error("Error parsing event: id={}", cloudEvent.getId(), e);
            return;
        }

        // Process the message
        handleMessage(cloudEvent, node);

    }

    /**
     * Handles an incoming notification message in CloudEvent format.
     *
     * @param cloudEvent The CloudEvent containing the notification message
     * @param data The parsed JSON data from the message
     */
    private void handleMessage(
            final CloudEvent cloudEvent,
            final JsonNode data) {

        log.debug("Processing event: id={}, handler={}", cloudEvent.getId(), name);

        final String resourceId = Optional.ofNullable(config.getResource())
                .map(field -> fieldExtractorRegistry.extractResourceId(field, data))
                .orElse(null);
        log.debug("Event resource id: {}", resourceId);

        final Map<String, String> placeholders = fieldExtractorRegistry.extractPlaceholders(
                config.getPlaceholders(),
                data);
        log.debug("Event placeholders: {}", placeholders);

        final Set<String> receivers = fieldExtractorRegistry.extractReceivers(
                config.getReceivers(),
                data);
        log.debug("Event receivers: {}", receivers);

        if (receivers.isEmpty()) {
            log.warn("No receivers found for event: id={}, handler={}", cloudEvent.getId(), name);
        } else {
            log.info(
                    "Creating notifications for event: id={}, handler={}, type={}, resourceId={}, receivers={}",
                    cloudEvent.getId(),
                    name,
                    config.getType(),
                    resourceId,
                    receivers.size());
            createNotifications(resourceId, placeholders, receivers);
        }

    }

    private void createNotifications(
            final String resourceId,
            final Map<String, String> placeholders,
            final Set<String> receivers) {

        final InputCreateUserNotificationDto request = InputCreateUserNotificationDto.builder()
                .type(config.getType())
                .referencedId(resourceId)
                .placeholderValues(placeholders)
                .users(new ArrayList<>(receivers))
                .build();

        notificationBsClient.create(request);

    }

}
