package com.finconsgroup.itserr.marketplace.notificationfeeder.bs.helper;

import com.finconsgroup.itserr.messaging.autoconfigure.MessagingStreamProperties;
import com.finconsgroup.itserr.messaging.client.MessagingFactory;
import com.finconsgroup.itserr.messaging.consumer.ConsumerRegistry;
import com.finconsgroup.itserr.messaging.consumer.MessageConsumer;
import com.finconsgroup.itserr.messaging.consumer.handler.ConsumerMessageHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;

/**
 * Helper class that manages messaging factory and consumer registry operations, providing functionality to create and register message consumers.
 */
@Slf4j
@Component
public class MessagingHelper {

    /** Factory responsible for creating messaging components and consumers */
    private final MessagingFactory messagingFactory;
    /** Registry that manages the lifecycle of message consumers */
    private final ConsumerRegistry consumerRegistry;

    /**
     * Constructs a MessagingHelper with the specified messaging factory map and consumer registry.
     *
     * @param messagingFactoryMap map of messaging factories keyed by their names
     * @param consumerRegistry registry for managing message consumer lifecycles
     */
    public MessagingHelper(
            final Map<String, MessagingFactory> messagingFactoryMap,
            final ConsumerRegistry consumerRegistry) {
        this.messagingFactory = getMessagingFactory(messagingFactoryMap);
        this.consumerRegistry = consumerRegistry;
    }

    @NonNull
    private static MessagingFactory getMessagingFactory(
            @NonNull final Map<String, MessagingFactory> messagingFactoryMap) {

        if (messagingFactoryMap.isEmpty()) {
            throw new UnsupportedOperationException("A messaging client must be provided in messaging.clients configuration property");
        }

        // TODO: handle multiple messaging clients
        if (messagingFactoryMap.size() > 1) {
            throw new UnsupportedOperationException("Only one messaging client is currently supported. Check your messaging.clients configuration property");
        }

        return messagingFactoryMap.values().iterator().next();

    }

    /**
     * Creates a list of message consumers for the specified messaging stream.
     *
     * @param messagingStreamProperties the properties defining the messaging stream, including consumer configuration
     * @param consumerHandlerMap a map of consumer handler implementations, keyed by handler name, used to process messages
     * @return a list of configured {@link MessageConsumer} instances for consuming messages
     */
    public List<MessageConsumer> createConsumers(
            final MessagingStreamProperties messagingStreamProperties,
            final Map<String, ConsumerMessageHandler> consumerHandlerMap) {
        return messagingFactory.createConsumers(messagingStreamProperties, consumerHandlerMap);
    }

    /**
     * Registers a new {@link MessageConsumer} to the registry for lifecycle management.
     *
     * @param messageConsumer the message consumer to be registered; it must be non-null
     */
    public void register(@NonNull final MessageConsumer messageConsumer) {
        consumerRegistry.register(messageConsumer);
    }

}
