package com.finconsgroup.itserr.marketplace.notificationfeeder.bs.config.properties;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.util.Map;

/**
 * Configuration for a placeholder that supports optional value mappings and default values.
 * <p>
 * This class allows placeholders to be configured with a JSON path, optional
 * mappings to transform extracted values (e.g., "true" → "approved"), and
 * a default value to use when the extracted value is null or blank.
 * </p>
 */
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class PlaceholderConfig {

    /**
     * The JSON path expression to extract the value from the source data.
     * For example: {@code $.additionalData.approved}
     */
    @NonNull
    private String path;

    /**
     * Optional mappings to transform extracted values.
     * Keys are the original values, and values are the transformed values.
     * For example: {@code {"true": "approved", "false": "rejected"}}
     */
    @Nullable
    private Map<String, String> mappings;

    /**
     * Optional default value to use when the extracted value is null or blank.
     * For example: {@code "no message provided"}
     */
    @Nullable
    private String defaultValue;

    /**
     * Applies the configured mapping and default value logic to the given value.
     * <ol>
     *   <li>If value is null or blank and defaultValue is set, returns defaultValue</li>
     *   <li>If mappings are defined and contain the value, returns the mapped value</li>
     *   <li>Otherwise, returns the original value unchanged</li>
     * </ol>
     *
     * @param value the extracted value to transform
     * @return the transformed value according to the rules above
     */
    @Nullable
    public String applyMapping(@Nullable final String value) {
        // Handle null or blank values with default
        if (StringUtils.isBlank(value)) {
            return defaultValue;
        }
        // Apply mapping if exists
        if (mappings != null && mappings.containsKey(value)) {
            return mappings.get(value);
        }
        return value;
    }

}
