package com.finconsgroup.itserr.marketplace.search.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.search.dm.bean.SearchRequest;
import com.finconsgroup.itserr.marketplace.search.dm.config.CatalogSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.DefaultSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.SearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputCatalogDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputCatalogDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputCatalogFavouriteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputCatalogLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.entity.Catalog;
import com.finconsgroup.itserr.marketplace.search.dm.enums.Category;
import com.finconsgroup.itserr.marketplace.search.dm.mapper.CatalogMapper;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CatalogRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomAggregationRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomQueryRepository;
import com.finconsgroup.itserr.marketplace.search.dm.service.CatalogService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Default implementation of {@link CatalogService} to perform search and document related operations
 * by connecting to an OpenSearch instance.
 */
@Service
@Slf4j
public class DefaultCatalogService implements CatalogService {

    private final CatalogRepository catalogRepository;
    private final CatalogMapper catalogMapper;
    private final CatalogSearchProperties catalogSearchProperties;
    private final Map<String, String> sortFilterPropertyMap;

    public DefaultCatalogService(CatalogRepository catalogRepository,
                                 CatalogMapper catalogMapper,
                                 CatalogSearchProperties catalogSearchProperties,
                                 DefaultSearchProperties defaultSearchProperties) {
        this.catalogRepository = catalogRepository;
        this.catalogMapper = catalogMapper;
        this.catalogSearchProperties = catalogSearchProperties;
        this.sortFilterPropertyMap = buildSortFilterPropertyMap(catalogSearchProperties.search(),
            defaultSearchProperties.search().sortFilterPropertyMap());
    }

    @Override
    @Transactional
    @NonNull
    public OutputCatalogDto upsertDocument(@NonNull InputCatalogDto dto) {
        Catalog catalog = catalogMapper.toEntity(dto);
        Catalog savedCatalog = catalogRepository.save(catalog);
        return catalogMapper.toDto(savedCatalog);
    }

    @Override
    @Transactional(readOnly = true)
    @NonNull
    public OutputCatalogDto getDocument(@NonNull String id) {
        Catalog savedCatalog = catalogRepository
            .findById(id)
            .orElseThrow(() -> new WP2ResourceNotFoundException("search_dm_catalog_not_found"));
        return catalogMapper.toDto(savedCatalog);
    }

    @Override
    @Transactional
    public void deleteDocument(@NonNull String id) {
        if (!catalogRepository.existsById(id)) {
            throw new WP2ResourceNotFoundException("search_dm_catalog_not_found");
        }

        catalogRepository.deleteById(id);
    }

    @Override
    public void deleteAll() {
        if (!catalogSearchProperties.search().enableDeleteAll()) {
            throw new WP2BusinessException("search_dm_catalog_delete_all_not_enabled");
        }

        catalogRepository.deleteAll();
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        Page<OutputGlobalSearchAutoCompleteDataDto> resultPage = search(
            SearchRequest.builder().terms(terms).build(),
            catalogSearchProperties.search().autoCompletion().sourceFields(),
            catalogMapper::toAutoCompleteDataDto,
            PageRequest.of(0, catalogSearchProperties.search().autoCompletion().topHitsLimit())
        );
        if (resultPage.isEmpty()) {
            return List.of();
        } else {
            return List.of(OutputGlobalSearchAutoCompleteDto
                .builder()
                .category(Category.CATALOG.getId())
                .data(resultPage.getContent())
                .build());
        }
    }


    @NonNull
    @Override
    @Transactional(readOnly = true)
    public Page<OutputCatalogLocalSearchDto> getLocalSearch(String terms, String filters, @NonNull Pageable pageable) {
        SearchRequest searchRequest = SearchRequest
            .builder()
            .terms(terms)
            .queryFilters(buildQueryFilters(filters, sortFilterPropertyMap))
            .build();
        Pageable sortedPageable = applySort(pageable, catalogSearchProperties.search(), sortFilterPropertyMap);
        return search(searchRequest,
            catalogSearchProperties.search().local().sourceFields(),
            catalogMapper::toLocalSearchDto,
            sortedPageable);
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchDto> getSearch(@NonNull String terms) {
        Map<String, List<OutputGlobalSearchDataDto>> searchResultAggregatedByType = searchByAggregation(
            SearchRequest.builder().terms(terms).build(),
            catalogSearchProperties.search().globalSearch().sourceFields(),
            catalogSearchProperties.search().globalSearch().topHitsLimit(),
            catalogSearchProperties.search().globalSearch().aggregation(),
            catalogMapper::toGlobalSearchDataDto);

        if (searchResultAggregatedByType.isEmpty()) {
            return List.of();
        } else {
            List<OutputGlobalSearchDto> globalSearchResults = new ArrayList<>();
            searchResultAggregatedByType.forEach((type, results) ->
                globalSearchResults.add(OutputGlobalSearchDto
                    .builder()
                    .category(Category.CATALOG.getId())
                    .type(type)
                    .data(results)
                    .build()));
            return globalSearchResults;
        }
    }

    @NonNull
    @Override
    @Transactional(readOnly = true)
    public Page<OutputCatalogFavouriteDto> getFavourite(@NonNull List<String> ids, String filters, @NonNull Pageable pageable) {
        SearchRequest searchRequest = SearchRequest
            .builder()
            .ids(ids)
            .queryFilters(buildQueryFilters(filters, sortFilterPropertyMap))
            .build();
        Pageable sortedPageable = applySort(pageable, catalogSearchProperties.search(), sortFilterPropertyMap);
        return search(searchRequest,
            catalogSearchProperties.search().favourite().sourceFields(),
            catalogMapper::toFavouriteDto,
            sortedPageable);
    }

    @NonNull
    @Override
    public Class<Catalog> getDocumentClass() {
        return Catalog.class;
    }

    @NonNull
    @Override
    public SearchProperties getSearchProperties() {
        return catalogSearchProperties.search();
    }

    @NonNull
    @Override
    public CustomQueryRepository getCustomQueryRepository() {
        return catalogRepository;
    }

    @NonNull
    @Override
    public CustomAggregationRepository getCustomAggregationRepository() {
        return catalogRepository;
    }
}
