package com.finconsgroup.itserr.marketplace.search.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.bean.QueryFilter;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.search.dm.bean.SearchRequest;
import com.finconsgroup.itserr.marketplace.search.dm.config.DefaultSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.InstitutionalPageSearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.config.SearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.constant.SecurityConstants;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDataDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputInstitutionalPageLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.entity.InstitutionalPage;
import com.finconsgroup.itserr.marketplace.search.dm.entity.UserProfile;
import com.finconsgroup.itserr.marketplace.search.dm.entity.UserProfileMinimal;
import com.finconsgroup.itserr.marketplace.search.dm.enums.Category;
import com.finconsgroup.itserr.marketplace.search.dm.event.UserProfileUpdatedEvent;
import com.finconsgroup.itserr.marketplace.search.dm.mapper.InstitutionalPageMapper;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomAggregationRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.CustomQueryRepository;
import com.finconsgroup.itserr.marketplace.search.dm.repository.InstitutionalPageRepository;
import com.finconsgroup.itserr.marketplace.search.dm.service.InstitutionalPageService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationListener;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.function.Function;

/**
 * Default implementation of {@link InstitutionalPageService} to perform search and document related operations
 * by connecting to an OpenSearch instance.
 */
@Service
@Slf4j
public class DefaultInstitutionalPageService implements InstitutionalPageService, ApplicationListener<UserProfileUpdatedEvent> {

    private final InstitutionalPageRepository institutionalPageRepository;
    private final InstitutionalPageMapper institutionalPageMapper;
    private final InstitutionalPageSearchProperties institutionalPageSearchProperties;
    private final Map<String, String> sortFilterPropertyMap;

    public DefaultInstitutionalPageService(InstitutionalPageRepository institutionalPageRepository,
                                           InstitutionalPageMapper institutionalPageMapper,
                                           InstitutionalPageSearchProperties institutionalPageSearchProperties,
                                           DefaultSearchProperties defaultSearchProperties) {
        this.institutionalPageRepository = institutionalPageRepository;
        this.institutionalPageMapper = institutionalPageMapper;
        this.institutionalPageSearchProperties = institutionalPageSearchProperties;
        this.sortFilterPropertyMap = buildSortFilterPropertyMap(institutionalPageSearchProperties.search(),
                defaultSearchProperties.search().sortFilterPropertyMap());
    }

    @Override
    @Transactional
    @NonNull
    public OutputInstitutionalPageDto upsertDocument(@NonNull InputInstitutionalPageDto dto) {
        InstitutionalPage institutionalPage = institutionalPageMapper.toEntity(dto);
        InstitutionalPage savedInstitutionalPage = institutionalPageRepository.save(institutionalPage);
        return institutionalPageMapper.toDto(savedInstitutionalPage);
    }

    @Override
    @Transactional(readOnly = true)
    @NonNull
    public OutputInstitutionalPageDto getDocument(@NonNull String id) {
        InstitutionalPage savedInstitutionalPage = institutionalPageRepository
                .findById(id)
                .orElseThrow(() -> new WP2ResourceNotFoundException(
                        "search_dm_institutionalPage_not_found"));
        return institutionalPageMapper.toDto(savedInstitutionalPage);
    }

    @Override
    @Transactional
    public void deleteDocument(@NonNull String id) {
        if (!institutionalPageRepository.existsById(id)) {
            throw new WP2ResourceNotFoundException("search_dm_institutionalPage_not_found");
        }

        institutionalPageRepository.deleteById(id);
    }

    @Override
    public void deleteAll() {
        if (!institutionalPageSearchProperties.search().enableDeleteAll()) {
            throw new WP2BusinessException("search_dm_institutionalPage_delete_all_not_enabled");
        }

        institutionalPageRepository.deleteAll();
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        Page<OutputGlobalSearchAutoCompleteDataDto> resultPage = search(
                SearchRequest.builder().terms(terms).queryFilters(addDefaultFilters(List.of())).build(),
                institutionalPageSearchProperties.search().autoCompletion().sourceFields(),
                institutionalPageMapper::toAutoCompleteDataDto,
                PageRequest.of(0, institutionalPageSearchProperties.search().autoCompletion().topHitsLimit())
        );
        if (resultPage.isEmpty()) {
            return List.of();
        } else {
            return List.of(OutputGlobalSearchAutoCompleteDto
                    .builder()
                    .category(Category.INSTITUTIONAL_PAGE.getId())
                    .data(resultPage.getContent())
                    .build());
        }
    }


    @NonNull
    @Override
    @Transactional(readOnly = true)
    public Page<OutputInstitutionalPageLocalSearchDto> getLocalSearch(String terms, String filters,
                                                                      @NonNull Pageable pageable) {
        SearchRequest searchRequest = SearchRequest
                .builder()
                .terms(terms)
                .queryFilters(addDefaultFilters(buildQueryFilters(filters, sortFilterPropertyMap)))
                .build();
        Pageable sortedPageable = applySort(pageable, institutionalPageSearchProperties.search(),
                sortFilterPropertyMap);
        return search(searchRequest,
                institutionalPageSearchProperties.search().local().sourceFields(),
                institutionalPageMapper::toLocalSearchDto,
                sortedPageable);
    }

    @Override
    @NonNull
    @Transactional(readOnly = true)
    public List<OutputGlobalSearchDto> getSearch(@NonNull String terms) {
        Map<String, List<OutputGlobalSearchDataDto>> searchResultAggregatedByType = searchByAggregation(
                SearchRequest.builder().terms(terms).queryFilters(addDefaultFilters(List.of())).build(),
                institutionalPageSearchProperties.search().globalSearch().sourceFields(),
                institutionalPageSearchProperties.search().globalSearch().topHitsLimit(),
                institutionalPageSearchProperties.search().globalSearch().aggregation(),
                institutionalPageMapper::toGlobalSearchDataDto);

        if (searchResultAggregatedByType.isEmpty()) {
            return List.of();
        } else {
            List<OutputGlobalSearchDto> globalSearchResults = new ArrayList<>();
            searchResultAggregatedByType.forEach((type, results) ->
                    globalSearchResults.add(OutputGlobalSearchDto
                            .builder()
                            .category(Category.INSTITUTIONAL_PAGE.getId())
                            .type(type)
                            .data(results)
                            .build()));
            return globalSearchResults;
        }
    }

    @Override
    public void onApplicationEvent(@NonNull UserProfileUpdatedEvent event) {
        try {
            if (event.isUserProfileMinimalUpdated()) {
                syncDocumentsForUserProfile(event.getUserProfile());
            }
        } catch (Exception ex) {
            log.error("Error occurred while syncing institutional pages for user profile with id - {}",
                    event.getUserProfile().getId(), ex);
        }
    }

    @NonNull
    @Override
    public Page<InstitutionalPage> searchDocumentsForUserProfileSync(@NonNull SearchRequest searchRequest, @NonNull Pageable pageable) {
        return search(searchRequest, null, Function.identity(), pageable);
    }

    @Override
    public void syncDocumentForUserProfile(@NonNull InstitutionalPage institutionalPage,
                                           @NonNull UserProfile userProfile) {
        boolean updated = false;
        if (institutionalPage.getMembers() != null) {
            for (UserProfileMinimal contributor : institutionalPage.getMembers()) {
                if (checkAndUpdateUserProfileMinimal(contributor, userProfile)) {
                    updated = true;
                }
            }
        }
        if (institutionalPage.getMaintainer() != null &&
                checkAndUpdateUserProfileMinimal(institutionalPage.getMaintainer(), userProfile)) {
            updated = true;
        }

        if (institutionalPage.getLastModifiedBy() != null &&
                checkAndUpdateUserProfileMinimal(institutionalPage.getLastModifiedBy(), userProfile)) {
            updated = true;
        }

        if (institutionalPage.getWpLeads() != null) {
            for (UserProfileMinimal wpLead : institutionalPage.getWpLeads()) {
                if (checkAndUpdateUserProfileMinimal(wpLead, userProfile)) {
                    updated = true;
                }
            }
        }

        if (updated) {
            institutionalPageRepository.save(institutionalPage);
        }
    }

    @NonNull
    @Override
    public Class<InstitutionalPage> getDocumentClass() {
        return InstitutionalPage.class;
    }

    @NonNull
    @Override
    public SearchProperties getSearchProperties() {
        return institutionalPageSearchProperties.search();
    }

    @NonNull
    @Override
    public CustomQueryRepository getCustomQueryRepository() {
        return institutionalPageRepository;
    }

    @NonNull
    @Override
    public CustomAggregationRepository getCustomAggregationRepository() {
        return institutionalPageRepository;
    }

    private List<QueryFilter> addDefaultFilters(@NonNull List<QueryFilter> queryFilters) {
        List<QueryFilter> filtersWithDefaults = new ArrayList<>(queryFilters);
        List<String> userIds = new ArrayList<>();
        // always fetch IPs that are public i.e. allowed for all users
        userIds.add(SecurityConstants.ALL_USERS);
        // if user has logged in, then additionally fetch IPs that are accessible to the logged-in user
        JwtTokenHolder.getUserId().map(UUID::toString).ifPresent(userIds::add);
        filtersWithDefaults.add(QueryFilter.builder().fieldName("allowedUserIds").filterValues(userIds).build());
        return filtersWithDefaults;
    }

    private boolean checkAndUpdateUserProfileMinimal(UserProfileMinimal userProfileMinimal, UserProfile userProfile) {
        if (Objects.equals(userProfileMinimal.getId(), userProfile.getId()) &&
                isUserProfileMinimalUpdated(userProfileMinimal, userProfile)) {
            updateUserProfileMinimal(userProfileMinimal, userProfile);
            return true;
        }
        return false;
    }

}
