package com.finconsgroup.itserr.marketplace.userprofile.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.OneToMany;
import jakarta.persistence.OrderBy;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.ToString;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.List;
import java.util.UUID;

@Entity
@Table(name = "expertise")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ExpertiseEntity extends AbstractUUIDEntity {

    @EqualsAndHashCode.Exclude
    @ToString.Exclude
    @ManyToOne
    @JoinColumn(name = "user_profile_id", nullable = false)
    private UserProfileEntity userProfile;

    /**
     * The unique identifier of the label.
     */
    @Column(name = "label_id")
    private UUID labelId;

    /**
     * The display name or name of the expertise.
     */
    @Column(name = "display_name")
    private String displayName;

    /**
     * The total number of endorsements received for this expertise.
     */
    @Column(name = "endorsement_count")
    private Integer endorsementCount;

    /**
     * The list of endorsements associated with this expertise.
     */
    @OneToMany(cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "expertise")
    @OrderBy("creationTime ASC")
    @ToString.Exclude
    private List<EndorsementEntity> endorsements;

    /**
     * The ordering column.
     */
    @Column(name = "expertise_order", nullable = false)
    private long expertiseOrder;


    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     * </p>
     */
    @Version
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now();
    }

    /**
     * Increments the endorsement count for this expertise.
     * <p>
     * Used when a new endorsement is received.
     * </p>
     */
    public void incrementEndorsements() {
        if (endorsementCount == null) {
            endorsementCount = 0;
        }
        endorsementCount++;
    }

    /**
     * Decrements the endorsement count if it's greater than zero.
     * <p>
     * Used when an endorsement is removed or withdrawn.
     * </p>
     */
    public void decrementEndorsements() {
        if (endorsementCount > 0) {
            endorsementCount--;
        }
    }

}
