package com.finconsgroup.itserr.marketplace.userprofile.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.enums.ThemeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.EnumType;
import jakarta.persistence.Enumerated;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.OneToOne;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.Instant;

/**
 * Entity representing user preferences in the User Profile service.
 * <p>
 * This entity stores user-specific display settings and metadata such as
 * the selected {@link ThemeType}, creation time, and last update time.
 * It maps to the {@code user_preference} table in the database.
 * </p>
 *
 * <p><b>Example usage:</b></p>
 * <pre>{@code
 * UserPreferenceEntity preference = UserPreferenceEntity.builder()
 *     .theme(ThemeType.SYSTEM)
 *     .build();
 * }</pre>
 */
@Entity
@Table(name = "user_preference")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class UserPreferenceEntity extends AbstractUUIDEntity {

    @EqualsAndHashCode.Exclude
    @OneToOne
    @JoinColumn(name = "user_profile_id", nullable = false)
    private UserProfileEntity userProfile;

    @Enumerated(EnumType.STRING)
    @Column(name = "theme", nullable = false)
    private ThemeType theme;

    /**
     * The timestamp when the userprofile was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the userprofile was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     * </p>
     */
    @Version
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now();
    }

}
