package com.finconsgroup.itserr.marketplace.userprofile.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.converter.StringListConverter;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Convert;
import jakarta.persistence.Entity;
import jakarta.persistence.Lob;
import jakarta.persistence.OneToMany;
import jakarta.persistence.OneToOne;
import jakarta.persistence.OrderColumn;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.ToString;
import lombok.experimental.SuperBuilder;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.UUID;

/**
 * Entity class representing a userprofile in the user-profile service.
 * This entity is mapped to the "user_profile" table in the database
 * and stores information such as
 * the userprofile name, description, associated category, creation time, and update time.
 *
 * <p>Example usage:
 * <pre>
 * UserProfileEntity userprofileEntity = UserProfileEntity.builder()
 *     .name("Sunset Valley Lot")
 *     .category(category)
 *     .description("Scenic rural land")
 *     .build();
 * </pre>
 * </p>
 */
@Entity
@Table(name = "user_profile")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class UserProfileEntity extends AbstractUUIDEntity {

    @Column(name = "first_name", nullable = false)
    private String firstName;

    @Column(name = "last_name", nullable = false)
    private String lastName;

    @Column(name = "email", nullable = false)
    private String email;

    @Column(name = "preferred_username", nullable = false)
    private String preferredUsername;

    @Column(name = "profile_folder_id", nullable = false)
    private UUID userProfileFolderId;

    @Column(name = "image_url")
    private String imageUrl;

    @Column(name = "orcid")
    private String orcid;

    @Column(name = "open_to_collaboration", nullable = false)
    private boolean openToCollaboration;

    @Column(name = "show_public_email", nullable = false)
    private boolean showPublicEmail;

    @Column(name = "personal_website")
    private String personalWebsite;

    @Column(name = "linkedin_profile")
    private String linkedinProfile;

    @Column(name = "social_profile")
    private String socialProfile;

    @Column(name = "short_bio", length = 1000)
    private String shortBio;

    @Column(name = "organization_affiliation", nullable = false)
    private String organizationAffiliation;

    @Lob
    @Convert(converter = StringListConverter.class)
    @Column(name = "ssd")
    private List<String> ssd;

    @OneToMany(mappedBy = "userProfile", cascade = CascadeType.ALL, orphanRemoval = true)
    @OrderColumn(name = "expertise_order")
    @ToString.Exclude
    private List<ExpertiseEntity> expertises;

    @OneToOne(mappedBy = "userProfile", cascade = CascadeType.ALL, orphanRemoval = true)
    private CVEntity cv;

    @OneToOne(mappedBy = "userProfile", cascade = CascadeType.ALL, orphanRemoval = true)
    private WorkingLocationEntity workingLocation;

    @OneToOne(mappedBy = "userProfile", cascade = CascadeType.ALL, orphanRemoval = true)
    private ReferenceEntity references;

    @OneToMany(mappedBy = "userProfile", cascade = CascadeType.ALL, orphanRemoval = true)
    private List<ProjectEntity> projects;

    @Column(name = "public_profile", nullable = false)
    private boolean publicProfile;

    @Lob
    @Convert(converter = StringListConverter.class)
    @Column(name = "interests")
    private List<String> interests;

    @OneToOne(mappedBy = "userProfile", cascade = CascadeType.ALL, orphanRemoval = true)
    private UserPreferenceEntity preferences;

    @Column(name = "hide_panel", nullable = false)
    private boolean hidePanel;

    /**
     * The timestamp when the userprofile was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private ZonedDateTime creationTime;

    /**
     * The timestamp when the userprofile was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private ZonedDateTime updateTime;

    /**
     * The user preferred language.
     */
    @Column(name = "language", nullable = false)
    private String language;

    /**
     * The user email notification consent.
     */
    @Column(name = "email_consent", nullable = false)
    private boolean emailConsent;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     * </p>
     */
    @Version
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        ZonedDateTime now = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.MICROS);
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = ZonedDateTime.now(ZoneOffset.UTC).truncatedTo(ChronoUnit.MICROS);
    }

}
