/**
 * Global exception handler for REST controllers in the userprofile system.
 * This class handles custom exceptions related to userprofile and category operations
 * and returns appropriate HTTP error responses.
 *
 * <p>Example usage:
 * <pre>
 * When a UserProfileNotFoundException is thrown, this handler returns a 404 NOT FOUND response.
 * </pre>
 * </p>
 */
package com.finconsgroup.itserr.marketplace.userprofile.dm.exception;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.context.request.WebRequest;

import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.createResponseEntityError;
import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.logError;
import static org.springframework.http.HttpStatus.CONFLICT;
import static org.springframework.http.HttpStatus.NOT_FOUND;

/**
 * Custom Exception Handler.
 */
@RestControllerAdvice
public class CustomExceptionHandler {

    /**
     * Handles UserProfileNotFoundException and returns a 404 NOT FOUND error response.
     *
     * @param ex      the UserProfileNotFoundException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 404
     */
    @ExceptionHandler({UserProfileNotFoundException.class})
    public ResponseEntity<ErrorResponseDto> handleUserProfileNotFoundException(
            final UserProfileNotFoundException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(NOT_FOUND, ex.getMessage());
    }

    /**
     * Handles UserProfileFoundException and returns a 409 FOUND error response.
     *
     * @param ex      the UserProfileFoundException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 409
     */
    @ExceptionHandler({UserProfileAlreadyExistsException.class})
    public ResponseEntity<ErrorResponseDto> handleUserProfileFoundException(
            final UserProfileAlreadyExistsException ex, final WebRequest request
    ) {
        logError(ex, request);
        return createResponseEntityError(CONFLICT, ex.getMessage());
    }
}
