package com.finconsgroup.itserr.marketplace.userprofile.dm.repository;

import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.UserProfileEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.mapper.UserProfileMapper;
import jakarta.validation.constraints.NotEmpty;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.List;
import java.util.UUID;

/**
 * {@link UserProfileRepository} repository.
 */
@Repository
public interface UserProfileRepository extends JpaRepository<UserProfileEntity, UUID> {

    @NonNull
    Page<UserProfileEntity> findAllByIdIn(@NonNull Collection<UUID> ids, @NonNull Pageable pageable);

    @Query("SELECT NEW com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileAutoCompleteDto(" +
            "u.id, u.firstName, u.lastName, u.preferredUsername, u.email, u.imageUrl) " +
            "FROM UserProfileEntity u WHERE " +
            "LOWER(u.preferredUsername) LIKE LOWER(CONCAT('%', CONCAT(:term, '%'))) OR " +
            "LOWER(u.firstName) LIKE LOWER(CONCAT('%', CONCAT(:term, '%'))) OR " +
            "LOWER(u.lastName) LIKE LOWER(CONCAT('%', CONCAT(:term, '%')))")
    Page<OutputUserProfileAutoCompleteDto> findUsersForAutocomplete(@Param("term") String term, @NonNull Pageable pageable);

    @NonNull
    Page<UserProfileEntity>  findAllByPreferredUsernameIn(@NotEmpty Collection<String> principals, Pageable pageable);

    @Query("SELECT up FROM UserProfileEntity up " +
            "WHERE CAST(up.id AS string) IN :tokenInfos " +
            "OR up.email IN :tokenInfos " +
            "OR up.preferredUsername IN :tokenInfos")
    List<UserProfileEntity> findByMixedTokenInfo(@Param("tokenInfos") List<String> tokenInfos);

    @Query(value = """
              with strings_to_check as (
                select
                  lower(unnest(string_to_array(:stringsToCheck, :delimiter))) as str),
              up_interests as (
                select
                  up.id,
                  lower(unnest(string_to_array(convert_from(lo_get(up.interests::oid), 'UTF-8'), :interestsDelimiter))) as interests
                from
                  user_profile up
                where up.public_profile = true
              )
                select
                  distinct up_interests.id
                from
                  up_interests
                cross join strings_to_check
                where
                  /* strings_to_check.str like '%' || interests || '%'  -- contains check */
                  regexp_like(strings_to_check.str, '\\y' || interests || '\\y', 'i') /* words check */
                order by up_interests.id
            """, nativeQuery = true)
    List<UUID> findMatchingInterests(@Param("stringsToCheck") String stringsToCheck,
                                     @Param("delimiter") String delimiter,
                                     @Param("interestsDelimiter") String interestsDelimiter);

}