package com.finconsgroup.itserr.marketplace.userprofile.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ValidationException;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputEndorsementAcknowledgementDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.ArchivedEndorsementEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.ArchivedExpertiseEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.EndorsementEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.ExpertiseEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.mapper.ArchivedEndorsementMapper;
import com.finconsgroup.itserr.marketplace.userprofile.dm.mapper.EndorsementMapper;
import com.finconsgroup.itserr.marketplace.userprofile.dm.repository.ArchivedEndorsementRepository;
import com.finconsgroup.itserr.marketplace.userprofile.dm.repository.EndorsementRepository;
import com.finconsgroup.itserr.marketplace.userprofile.dm.repository.ExpertiseRepository;
import com.finconsgroup.itserr.marketplace.userprofile.dm.service.EndorsementService;
import com.finconsgroup.itserr.marketplace.userprofile.dm.service.UserProfileService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.UUID;

/**
 * Default implementation of {@link EndorsementService}
 * to perform operations related to userprofile resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultEndorsementService implements EndorsementService {

    private static final String EXPERTISE_USER_BUSINESS_KEY = "{expertiseId: %s, userId: %s}";
    private static final String USER_PROFILE_BUSINESS_KEY = "{user cannot endorse their own expertise. userId: %s, profileId: %s}";

    private final ExpertiseRepository expertiseRepository;

    private final EndorsementRepository endorsementRepository;
    private final EndorsementMapper endorsementMapper;

    private final ArchivedEndorsementRepository archivedEndorsementRepository;
    private final ArchivedEndorsementMapper archivedEndorsementMapper;

    private final UserProfileService userProfileService;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputEndorsementAcknowledgementDto addEndorsement(@NonNull UUID userId,
                                                              @NonNull UUID profileId,
                                                              @NonNull UUID expertiseId) {

        if (userId.equals(profileId)) {
            throw new WP2ValidationException(USER_PROFILE_BUSINESS_KEY.formatted(userId, profileId));
        }

        // check if the user had already endorsed the expertise
        if (endorsementRepository.existsByExpertiseIdAndEndorserId(expertiseId, userId)) {
            throw new WP2DuplicateResourceException(EXPERTISE_USER_BUSINESS_KEY.formatted(expertiseId, userId));
        }

        // check if expertise/skill exist for the given profileId
        ExpertiseEntity existingExpertiseEntity = findByIdAndUserIdOrThrow(expertiseId, profileId);

        EndorsementEntity endorsementEntity = EndorsementEntity.builder()
                .expertise(existingExpertiseEntity)
                .expertiseId(expertiseId)
                .endorserId(userId)
                .endorsementOrder(existingExpertiseEntity.getEndorsementCount())
                .build();

        EndorsementEntity savedEndorsementEntity = endorsementRepository.saveAndFlush(endorsementEntity);

        existingExpertiseEntity.incrementEndorsements();
        expertiseRepository.saveAndFlush(existingExpertiseEntity);

        OutputUserProfileDto outputUserProfileDto = userProfileService.getById(userId);
        return endorsementMapper.toEndorsementAcknowledgementDto(savedEndorsementEntity, outputUserProfileDto,
                profileId, existingExpertiseEntity.getDisplayName());
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void removeEndorsement(@NonNull UUID userId, @NonNull UUID profileId, @NonNull UUID expertiseId) {

        EndorsementEntity endorsementEntity = endorsementRepository
                .findByExpertiseIdAndEndorserId(expertiseId, userId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(EXPERTISE_USER_BUSINESS_KEY.formatted(expertiseId, userId)));
        endorsementRepository.delete(endorsementEntity);

        ArchivedEndorsementEntity archivedEndorsementEntity = archivedEndorsementMapper.toArchivedEntity(endorsementEntity);
        archivedEndorsementEntity.setExpertise(ArchivedExpertiseEntity.builder().id(expertiseId).build());
        archivedEndorsementRepository.save(archivedEndorsementEntity);

        ExpertiseEntity existingExpertiseEntity = findByIdAndUserIdOrThrow(expertiseId, profileId);
        existingExpertiseEntity.decrementEndorsements();
        expertiseRepository.saveAndFlush(existingExpertiseEntity);
    }

    private ExpertiseEntity findByIdAndUserIdOrThrow(@NonNull UUID expertiseId, @NonNull UUID profileId) {
        return expertiseRepository
                .findByIdAndUserProfileId(expertiseId, profileId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(EXPERTISE_USER_BUSINESS_KEY.formatted(expertiseId, profileId)));
    }
}