package com.finconsgroup.itserr.marketplace.userprofile.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.AuthenticatedUser;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputAddProjectToUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputExpertiseDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByIdsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByPrincipalsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByTokenInfoDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputPatchUserProfileProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputRemoveProjectFromUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputUpdateUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputUserProfileStatusChangeDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputEndorsementDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileFolderDetailsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.ArchivedExpertiseEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.ExpertiseEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.ProjectEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.entity.UserProfileEntity;
import com.finconsgroup.itserr.marketplace.userprofile.dm.exception.UserProfileAlreadyExistsException;
import com.finconsgroup.itserr.marketplace.userprofile.dm.exception.UserProfileNotFoundException;
import com.finconsgroup.itserr.marketplace.userprofile.dm.mapper.ArchivedExpertiseMapper;
import com.finconsgroup.itserr.marketplace.userprofile.dm.mapper.ExpertiseMapper;
import com.finconsgroup.itserr.marketplace.userprofile.dm.mapper.ProjectMapper;
import com.finconsgroup.itserr.marketplace.userprofile.dm.mapper.UserProfileMapper;
import com.finconsgroup.itserr.marketplace.userprofile.dm.repository.ArchivedExpertiseRepository;
import com.finconsgroup.itserr.marketplace.userprofile.dm.repository.ProjectRepository;
import com.finconsgroup.itserr.marketplace.userprofile.dm.repository.UserProfileRepository;
import com.finconsgroup.itserr.marketplace.userprofile.dm.service.UserProfileService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import static com.finconsgroup.itserr.marketplace.userprofile.dm.util.Constants.STRING_LIST_SEPARATOR;

/**
 * Default implementation of {@link UserProfileService}
 * to perform operations related to userprofile resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultUserProfileService implements UserProfileService {

    private final String ID_SEPARATOR = "-";
    private final String INTERESTS_CHECK_DELIMITER = "~^~";

    private final UserProfileRepository userProfileRepository;
    private final UserProfileMapper userProfileMapper;
    private final ProjectRepository projectRepository;
    private final ProjectMapper projectMapper;

    private final ExpertiseMapper expertiseMapper;

    private final ArchivedExpertiseRepository archivedExpertiseRepository;
    private final ArchivedExpertiseMapper archivedExpertiseMapper;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputUserProfileDto create(@NonNull AuthenticatedUser authenticatedUser, @NonNull InputUserProfileDto inputUserProfileDto) {
        if (userProfileRepository.existsById(authenticatedUser.userId())) {
            throw new UserProfileAlreadyExistsException(authenticatedUser.userId());
        }

        UserProfileEntity userProfileEntity = userProfileMapper.toEntity(inputUserProfileDto, authenticatedUser.userId());
        userProfileEntity.setId(authenticatedUser.userId());
        userProfileEntity.setFirstName(authenticatedUser.firstName());
        userProfileEntity.setLastName(authenticatedUser.lastName());
        userProfileEntity.setEmail(authenticatedUser.email());
        userProfileEntity.setPreferredUsername(authenticatedUser.preferredUsername());
        userProfileEntity.setHidePanel(false);

        updateAssociationsForSave(userProfileEntity);
        UserProfileEntity savedUserProfileEntity = userProfileRepository.saveAndFlush(userProfileEntity);

        OutputUserProfileDto outputUserProfileDto = userProfileMapper.toDto(savedUserProfileEntity);
        enrichUserProfileExpertise(outputUserProfileDto);
        return outputUserProfileDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputUserProfileDto> findAll(@NonNull Pageable pageable) {
        return userProfileRepository.findAll(pageable)
                .map(userProfileMapper::toDto);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputUserProfileAutoCompleteDto> getAutoCompletions(@NonNull String terms, @NonNull Pageable pageable) {
        return userProfileRepository.findUsersForAutocomplete(terms, pageable);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputUserProfileDto findById(@NotNull AuthenticatedUser authenticatedUser) {
        return userProfileRepository.findById(authenticatedUser.userId())
                .map(userProfileEntity -> {
                    OutputUserProfileDto userProfile = userProfileMapper.toDto(userProfileEntity);
                    userProfile.setUserExists(true);
                    enrichUserProfileExpertise(userProfile);
                    return userProfile;
                })
                .orElse(buildDefaultUserProfile(authenticatedUser));

    }

    @Override
    public @NotNull OutputUserProfileDto getById(@NotNull UUID profileId) {
        return userProfileRepository.findById(profileId)
                .map(userProfileEntity -> {
                    OutputUserProfileDto userProfile = userProfileMapper.toDto(userProfileEntity);
                    userProfile.setUserExists(true);
                    enrichUserProfileExpertise(userProfile);
                    return userProfile;
                })
                .orElseThrow(() -> new UserProfileNotFoundException(profileId));
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputUserProfileDto> findAllByIds(@NotNull InputFindUserProfilesByIdsDto inputFindUserProfilesByIdsDto,
                                                   @NonNull Pageable pageable) {
        return userProfileRepository
                .findAllByIdIn(inputFindUserProfilesByIdsDto.getIds(), pageable)
                .map(userProfileEntity -> {
                    OutputUserProfileDto userProfile = userProfileMapper.toDto(userProfileEntity);
                    userProfile.setUserExists(true);
                    return userProfile;
                });
    }

    private OutputUserProfileDto buildDefaultUserProfile(AuthenticatedUser authenticatedUser) {
        return OutputUserProfileDto.builder()
                .id(authenticatedUser.userId())
                .userExists(false)
                .firstName(authenticatedUser.firstName())
                .lastName(authenticatedUser.lastName())
                .email(authenticatedUser.email())
                .showPublicEmail(true)
                .hidePanel(false)
                .build();
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputUserProfileDto updateById(UUID userId, @NonNull InputUpdateUserProfileDto inputUpdateUserProfileDto) {
        UserProfileEntity userprofileEntity = userProfileRepository.findById(userId)
                .orElseThrow(() -> new UserProfileNotFoundException(userId));
        userProfileMapper.updateEntity(inputUpdateUserProfileDto, userprofileEntity);
        updateExpertise(inputUpdateUserProfileDto, userprofileEntity);
        updateAssociationsForSave(userprofileEntity);
        UserProfileEntity savedUserProfileEntity = userProfileRepository.save(userprofileEntity);
        OutputUserProfileDto outputUserProfileDto = userProfileMapper.toDto(savedUserProfileEntity);
        enrichUserProfileExpertise(outputUserProfileDto);
        return outputUserProfileDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputUserProfileFolderDetailsDto getUserProfileFolderId(UUID userId) {
        UserProfileEntity userProfileEntity = userProfileRepository.findById(userId)
                .orElseThrow(() -> new UserProfileNotFoundException(userId));
        return new OutputUserProfileFolderDetailsDto(userId, userProfileEntity.getUserProfileFolderId());
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, noRollbackFor = Exception.class)
    public OutputPatchUserProfileDto patchUserProfileInfo(InputPatchUserProfileDto patchUserProfileDto) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        UserProfileEntity userProfileEntity = userProfileRepository.findById(userId)
                .orElseThrow(() -> new UserProfileNotFoundException(userId));
        userProfileMapper.patchEntity(patchUserProfileDto, userProfileEntity);
        UserProfileEntity savedUserProfileEntity = userProfileRepository.save(userProfileEntity);
        return userProfileMapper.toPatchUserDto(savedUserProfileEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true)
    public Page<OutputUserProfileDto> findAllByPrincipals(InputFindUserProfilesByPrincipalsDto inputFindUserProfilesByPrincipalsDto, Pageable pageable) {
        if (inputFindUserProfilesByPrincipalsDto.getPrincipals() == null) {
            return Page.empty();
        }
        return userProfileRepository
                .findAllByPreferredUsernameIn(inputFindUserProfilesByPrincipalsDto.getPrincipals(), pageable)
                .map(userProfileEntity -> {
                    OutputUserProfileDto userProfile = userProfileMapper.toDto(userProfileEntity);
                    userProfile.setUserExists(true);
                    return userProfile;
                });
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public List<OutputUserProfileDto> addProjectToUserProfiles(@NonNull InputAddProjectToUserProfilesDto inputAddProjectToUserProfilesDto) {

        List<UserProfileEntity> userProfiles = userProfileRepository.findAllById(inputAddProjectToUserProfilesDto.getUserIds());
        InputProjectDto inputProjectDto = inputAddProjectToUserProfilesDto.getProject();

        if (userProfiles.isEmpty()) {
            return List.of();
        }

        List<ProjectEntity> existingProjects = projectRepository
                .findByUserProfileIdInAndProjectId(
                        inputAddProjectToUserProfilesDto.getUserIds(),
                        inputProjectDto.getProjectId()
                );

        Set<String> existingCombinations = existingProjects.stream()
                .map(project -> project.getUserProfile().getId() + ID_SEPARATOR + project.getProjectId())
                .collect(Collectors.toSet());

        List<ProjectEntity> newProjectEntities = new ArrayList<>();

        for (UserProfileEntity userProfile : userProfiles) {
            String combinationKey = userProfile.getId() + ID_SEPARATOR + inputProjectDto.getProjectId();

            if (!existingCombinations.contains(combinationKey)) {
                ProjectEntity projectEntity = ProjectEntity.builder()
                        .projectId(inputProjectDto.getProjectId())
                        .displayName(inputProjectDto.getDisplayName())
                        .wp(inputProjectDto.getWp())
                        .rootProjectId(inputProjectDto.getRootProjectId())
                        .rootProjectDisplayName(inputProjectDto.getRootProjectDisplayName())
                        .userProfile(userProfile)
                        .wpLead(inputProjectDto.getWpLead())
                        .build();
                newProjectEntities.add(projectEntity);
            }
        }

        if (!newProjectEntities.isEmpty()) {
            projectRepository.saveAllAndFlush(newProjectEntities);
        }

        List<UUID> updatedUserIds = userProfiles.stream().map(AbstractUUIDEntity::getId).toList();
        userProfiles = userProfileRepository.findAllById(updatedUserIds);

        return userProfiles.stream()
                .map(userProfileMapper::toDto)
                .collect(Collectors.toList());
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public List<OutputUserProfileDto> removeProjectFromUserProfiles(@NonNull InputRemoveProjectFromUserProfilesDto inputRemoveProjectFromUserProfilesDto) {

        List<ProjectEntity> projectsToDelete = projectRepository
                .findByUserProfileIdInAndProjectId(
                        inputRemoveProjectFromUserProfilesDto.getUserIds(),
                        inputRemoveProjectFromUserProfilesDto.getProjectId()
                );

        projectRepository.deleteAll(projectsToDelete);
        projectRepository.flush();

        List<UserProfileEntity> userProfiles = userProfileRepository
                .findAllById(inputRemoveProjectFromUserProfilesDto.getUserIds());

        return userProfiles.stream()
                .map(userProfileMapper::toDto)
                .collect(Collectors.toList());
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public List<OutputUserProfileDto> patchUserProfileProject(@NonNull InputPatchUserProfileProjectDto inputPatchUserProfileProjectDto) {
        List<ProjectEntity> projectsToUpdate = projectRepository.findByUserProfileIdInAndProjectIdIn(inputPatchUserProfileProjectDto.getUserIds(),
                inputPatchUserProfileProjectDto.getProjectIdIds());

        if (projectsToUpdate.isEmpty()) {
            return List.of();
        }

        projectsToUpdate.forEach(projectEntity -> projectMapper.patchEntity(inputPatchUserProfileProjectDto, projectEntity));
        projectRepository.saveAllAndFlush(projectsToUpdate);

        List<UserProfileEntity> userProfiles = userProfileRepository
                .findAllById(inputPatchUserProfileProjectDto.getUserIds());

        return userProfiles.stream()
                .map(userProfileMapper::toDto)
                .collect(Collectors.toList());
    }

    @Override
    @Transactional(readOnly = true)
    public List<OutputUserProfileDto> findAllByTokenInfo(InputFindUserProfilesByTokenInfoDto inputFindUserProfilesByTokenInfoDto) {
        if (CollectionUtils.isEmpty(inputFindUserProfilesByTokenInfoDto.getTokenInfos())) {
            return List.of();
        }
        return userProfileRepository.findByMixedTokenInfo(inputFindUserProfilesByTokenInfoDto.getTokenInfos())
                .stream()
                .map(userProfileEntity -> {
                    OutputUserProfileDto userProfile = userProfileMapper.toDto(userProfileEntity);
                    userProfile.setUserExists(true);
                    return userProfile;
                }).toList();
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void processUserProfileStatusChange(@NonNull UUID userId, @NonNull Boolean active) {
        InputUserProfileStatusChangeDto inputUserProfileStatusChangeDto = buildInputUserProfileStatusChangeDto(active);
        UserProfileEntity userProfileEntity = userProfileRepository.findById(userId)
                .orElseThrow(() -> new UserProfileNotFoundException(userId));
        userProfileMapper.patchEntity(inputUserProfileStatusChangeDto, userProfileEntity);
        userProfileRepository.save(userProfileEntity);
    }

    @NonNull
    private InputUserProfileStatusChangeDto buildInputUserProfileStatusChangeDto(Boolean active) {
        InputUserProfileStatusChangeDto inputUserProfileStatusChangeDto = InputUserProfileStatusChangeDto.builder().build();
        // Activating or deactivating a user profile currently sets it to public or private respectively.
        inputUserProfileStatusChangeDto.setPublicProfile(active);
        return inputUserProfileStatusChangeDto;
    }

    @Override
    @Transactional(readOnly = true)
    public List<UUID> findMatchingInterests(@NonNull List<String> stringsToCheck) {
        return userProfileRepository.findMatchingInterests(String.join(INTERESTS_CHECK_DELIMITER, stringsToCheck),
                INTERESTS_CHECK_DELIMITER, STRING_LIST_SEPARATOR);
    }

    private void updateAssociationsForSave(@NonNull UserProfileEntity userProfileEntity) {
        if (userProfileEntity.getExpertises() != null) {
            userProfileEntity.getExpertises().forEach(expertise -> {
                if (expertise.getEndorsementCount() == null) {
                    expertise.setEndorsementCount(0);
                }

                expertise.setUserProfile(userProfileEntity);
            });
        }
    }

    private void updateExpertise(InputUpdateUserProfileDto inputUpdateUserProfileDto, UserProfileEntity userProfileEntity) {

        List<InputExpertiseDto> inputExpertises = inputUpdateUserProfileDto.getExpertises();
        List<ExpertiseEntity> existingExpertises = userProfileEntity.getExpertises();

        if (inputExpertises == null) {
            inputExpertises = Collections.emptyList();
        }
        if (existingExpertises == null) {
            existingExpertises = new ArrayList<>();
            userProfileEntity.setExpertises(existingExpertises);
        }

        Map<UUID, ExpertiseEntity> existingByLabelId = existingExpertises.stream()
                .filter(e -> e.getLabelId() != null)
                .collect(Collectors.toMap(ExpertiseEntity::getLabelId, e -> e));

        Set<UUID> inputLabelIds = inputExpertises.stream()
                .map(InputExpertiseDto::getLabelId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        // Update or add new expertises
        upsertExpertises(inputExpertises, existingByLabelId, existingExpertises);

        // Handle removals (archive + delete)
        removeDeletedExpertises(existingExpertises, inputLabelIds);
    }

    private void upsertExpertises(List<InputExpertiseDto> inputExpertises,
                                  Map<UUID, ExpertiseEntity> existingByLabelId,
                                  List<ExpertiseEntity> existingExpertises) {

        for (InputExpertiseDto inputDto : inputExpertises) {
            UUID labelId = inputDto.getLabelId();
            if (labelId == null) continue;

            ExpertiseEntity existingEntity = existingByLabelId.get(labelId);

            if (existingEntity != null) {
                // Update existing
                expertiseMapper.updateEntity(inputDto, existingEntity);
            } else {
                // Add new
                ExpertiseEntity newEntity = expertiseMapper.toEntity(inputDto);
                existingExpertises.add(newEntity);
            }
        }
    }

    private void removeDeletedExpertises(List<ExpertiseEntity> existingExpertises, Set<UUID> inputLabelIds) {

        Iterator<ExpertiseEntity> iterator = existingExpertises.iterator();

        while (iterator.hasNext()) {
            ExpertiseEntity expertiseEntity = iterator.next();
            UUID labelId = expertiseEntity.getLabelId();

            if (labelId == null || inputLabelIds.contains(labelId)) {
                continue;
            }

            // Archive endorsements
            archiveExpertiseWithEndorsements(expertiseEntity);

            // Remove from active profile
            iterator.remove();
        }
    }

    @Transactional
    private void archiveExpertiseWithEndorsements(ExpertiseEntity expertiseEntity) {
        UUID expertiseId = expertiseEntity.getId();

        ArchivedExpertiseEntity archivedExpertise = archivedExpertiseMapper.toArchivedEntity(expertiseEntity);
        archivedExpertise.getEndorsements().forEach(
                archivedEndorsementEntity -> {
                    archivedEndorsementEntity.setExpertise(ArchivedExpertiseEntity.builder().id(expertiseId).build());
                }
        );
        archivedExpertiseRepository.save(archivedExpertise);
    }

    private void enrichUserProfileExpertise(OutputUserProfileDto outputUserProfileDto) {

        Set<UUID> endorserUserProfileIds = extractEndorserUserProfileIds(outputUserProfileDto);

        if (endorserUserProfileIds.isEmpty()) {
            return;
        }

        Map<UUID, OutputUserProfileDto> userProfileDtoById = getUserProfiles(endorserUserProfileIds);

        Optional.ofNullable(outputUserProfileDto.getExpertises())
                .orElse(List.of())
                .forEach(expertise ->
                        Optional.ofNullable(expertise.getEndorsements())
                                .orElse(List.of())
                                .forEach(endorsement -> {
                                    UUID endorserId = endorsement.getEndorserId();
                                    if (endorserId != null) {
                                        OutputUserProfileDto endorserProfile = userProfileDtoById.get(endorserId);
                                        if (endorserProfile != null) {
                                            endorsement.setFirstName(endorserProfile.getFirstName());
                                            endorsement.setLastName(endorserProfile.getLastName());
                                            endorsement.setEmail(endorserProfile.getEmail());
                                            endorsement.setImageUrl(endorserProfile.getImageUrl());
                                            endorsement.setPreferredUsername(endorserProfile.getPreferredUsername());
                                            endorsement.setShortBio(endorserProfile.getShortBio());
                                        }
                                    }
                                })
                );
    }

    private Set<UUID> extractEndorserUserProfileIds(OutputUserProfileDto outputUserProfileDto) {
        return Optional.ofNullable(outputUserProfileDto.getExpertises())
                .orElse(List.of())
                .stream()
                .flatMap(expertise -> Optional.ofNullable(expertise.getEndorsements())
                        .orElse(List.of())
                        .stream())
                .map(OutputEndorsementDto::getEndorserId)
                .collect(Collectors.toSet());
    }

    private Map<UUID, OutputUserProfileDto> getUserProfiles(Set<UUID> userProfileIds) {
        if (userProfileIds == null || userProfileIds.isEmpty()) {
            return Map.of();
        }

        Map<UUID, OutputUserProfileDto> userProfileDtoById = new HashMap<>();
        try {
            List<UserProfileEntity> entities = userProfileRepository.findAllById(userProfileIds);
            entities.forEach(entity -> {
                OutputUserProfileDto userProfileDto = userProfileMapper.toDto(entity);
                userProfileDtoById.put(userProfileDto.getId(), userProfileDto);
            });
        } catch (Exception e) {
            log.error("Error loading endorser user profiles for IDs {}", userProfileIds, e);
        }
        return userProfileDtoById;
    }

}
